# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Repository Overview

This repository contains the SQL Formatter skill for Claude Code. It provides comprehensive SQL formatting capabilities following Oracle Database 19 best practices, with 13 detailed formatting rules for creating consistent, readable SQL code.

## Core Architecture

### Skill Structure

The SQL Formatter skill uses the following structure:

1. **SKILL.md** (Required) - Main skill configuration with YAML frontmatter
   - Contains `name` and `description` fields that determine when Claude activates the skill
   - Uses imperative/infinitive form (verb-first instructions) throughout
   - Kept lean (<5k words), detailed info moved to references/

2. **references/** - Documentation loaded into context as needed
   - `sql-formatting-rules.md` contains complete 13-rule formatting specification
   - Keeps SKILL.md focused while providing detailed formatting rules

3. **examples/** - Example SQL files demonstrating formatting
   - `unformatted.sql` - SQL before formatting
   - `formatted.sql` - SQL after formatting
   - `complex-query.sql` - Complex queries showing all rules applied

### GitHub Actions Workflow

The repository includes `.github/workflows/release-skill.yml` which automatically:

- Extracts version from VERSION file
- Validates SKILL.md structure (frontmatter with name/description)
- Builds skill distribution ZIP
- Attaches to GitHub releases

## Development Commands

### Testing the Skill Locally

Install skill to Claude Code skills directory:

```bash
# Windows (Git Bash)
cp -r . "$USERPROFILE/.claude/skills/sql-formatter"

# Unix/Mac
cp -r . ~/.claude/skills/sql-formatter
```

Test activation by asking Claude questions that match the skill description.

### Markdown Linting

The repository uses markdownlint-cli2 with line length checks disabled:

```bash
# Configuration in .markdownlint-cli2.jsonc
# MD013 (line length) is disabled to prevent linter warnings on long lines
```

### Git Configuration

For new repositories, configure git user locally:

```bash
git config --local user.name "your-username"
git config --local user.email "your-email@users.noreply.github.com"
```

### Release Process

1. Update VERSION file with new version (e.g., `1.0.0`)
2. Commit and push: `git commit -m "Release v1.0.0" && git push`
3. Create release: `gh release create v1.0.0 --generate-notes`
4. GitHub Actions automatically builds and attaches skill ZIP

See `docs/tasks/release/how-to-release.md` for detailed instructions.

## Important Files

- **SKILL.md** - Main skill file with YAML frontmatter (name, description)
- **VERSION** - Single version number for releases (e.g., `1.1.0`)
- **README.md** - User-facing documentation and installation instructions
- **references/sql-formatting-rules.md** - Complete SQL formatting specification with 13 rules
- **examples/** - Example SQL files (unformatted, formatted, complex queries)
- **.markdownlint-cli2.jsonc** - Markdown linter config (MD013 disabled)
- **docs/tasks/release/how-to-release.md** - Release workflow documentation
- **docs/tasks/tests/how-to-test-skill.md** - SQL formatter-specific testing guide

## Skill Development Best Practices

### SKILL.md Description Field

The description field determines when Claude activates the skill. Make it:

- Specific about trigger words users might say
- Clear about scenarios when to use the skill
- Include file types, actions, or topics relevant to the skill
- Use third-person form: "This skill should be used when..."

Example (from this skill):

```yaml
---
name: sql-formatter
description: This skill should be used when the user asks to format SQL code, polish SQL queries, improve SQL readability, or work with .sql files. Use when queries mention SQL formatting, code beautification, Oracle SQL, or database query polishing.
---
```

### Writing Style

- Use imperative/infinitive form (verb-first instructions) throughout SKILL.md
  - Good: "Run the script to process files"
  - Bad: "You should run the script" or "You can run the script"
- Keep SKILL.md focused and concise (<5k words)
- Move detailed documentation to `references/` files
- Provide real examples, not hypothetical ones

### Markdown File Size Constraints

**CRITICAL**: Each markdown file packaged in a skill must adhere to strict size limits:

- **Preferred**: Keep each `.md` file under 500 lines
- **Maximum**: More than 1000 lines is NOT ALLOWED (critical constraint)

This applies to all markdown files in the skill:
- SKILL.md
- files in `references/` directory
- documentation files in `docs/`

If content exceeds these limits:
- Split into multiple smaller files
- Move detailed content to separate reference files
- Use clear cross-references between files

### SQL Formatter Skill Resources

This skill includes:

- **references/sql-formatting-rules.md** - Complete 13-rule specification with examples
  - Loaded when complex formatting scenarios require detailed rules
  - Contains all formatting patterns and edge cases

- **examples/** - Practical SQL examples
  - Used for testing and demonstrating formatting capabilities
  - Compare unformatted vs formatted to validate changes

### Testing Approach

Follow the framework in `docs/tasks/tests/how-to-test-skill.md`:

1. Install skill locally
2. Test activation with various phrasings
3. Verify core functionality
4. Test error handling
5. Validate documentation accuracy

Create specific test cases for each skill feature with expected inputs/outputs.

## Common Workflows

### Testing Changes Locally

1. Make changes to SKILL.md, references, or examples
2. Install to Claude Code: `cp -r . ~/.claude/skills/sql-formatter`
3. Test activation with SQL formatting requests
4. Verify formatting rules are applied correctly
5. Compare output with examples/formatted.sql

### Creating a New Release

1. Test all changes locally using docs/tasks/tests/how-to-test-skill.md
2. Update VERSION file with new version (e.g., `1.2.0`)
3. Commit changes: `git commit -m "Release v1.2.0: Description"`
4. Push to GitHub: `git push`
5. Create release: `gh release create v1.2.0 --generate-notes`
6. GitHub Actions automatically builds and attaches skill ZIP

See `docs/tasks/release/how-to-release.md` for detailed instructions.

### Validation Before Release

Required checks:

- SKILL.md has valid YAML frontmatter with `name` and `description`
- VERSION file exists with semantic version number
- README.md is customized (no template placeholders)
- Skill installs correctly to `~/.claude/skills/`
- Claude activates skill when expected
- All formatting rules work correctly
- Documentation matches actual behavior

## Troubleshooting

### Skill Doesn't Activate

1. Verify SKILL.md has valid YAML frontmatter: `head -10 SKILL.md`
2. Check description is specific with trigger words
3. Try explicit request: "Use the sql-formatter skill to format this SQL: [query]"
4. Restart Claude Code (skills loaded on startup)

### GitHub Actions Release Fails

Common issues:

- VERSION file missing or empty
- SKILL.md missing YAML frontmatter
- Invalid frontmatter (missing `name` or `description`)

Check workflow logs: `gh run view --log`

### Formatting Not Applied Correctly

**Symptoms:**
- SQL formatting differs from examples
- Some rules not applied consistently

**Checks:**

1. Compare with reference files: `examples/formatted.sql`
2. Review detailed rules: `references/sql-formatting-rules.md`
3. Test with simple queries first (basic SELECT)
4. Ensure input SQL is syntactically valid

## Documentation Structure

```text
docs/
├── guides/              # Additional documentation (optional)
│   └── README.md
└── tasks/
    ├── release/
    │   └── how-to-release.md    # Release workflow
    └── tests/
        └── how-to-test-skill.md # Testing framework
```

## Version Numbering

Follow semantic versioning (MAJOR.MINOR.PATCH):

- `0.0.1` - Initial development
- `0.1.0` - First feature complete
- `1.0.0` - First stable release
- `1.1.0` - New feature (backward compatible)
- `1.1.1` - Bug fix
- `2.0.0` - Breaking change

## References

- [Claude Code Skills Documentation](https://docs.claude.com/en/docs/claude-code/skills)
- [Skill Authoring Best Practices](https://docs.claude.com/en/docs/agents-and-tools/agent-skills/best-practices)
- [skill-creator Skill](https://github.com/anthropics/example-skills) - Official skill creation tool
