# better-auth v1.4.x New Features

New features added in v1.4.4 through v1.4.8.

---

## Background Tasks (v1.4.8)

Defer non-critical actions like email sending to improve response times.

### Configuration

```typescript
import { betterAuth } from "better-auth";

export const auth = betterAuth({
  // Global background tasks configuration
  backgroundTasks: {
    enabled: true,
    // Queue implementation (default: in-memory)
    queue: "memory",  // or "redis", "custom"
  },
  // ...
});
```

### Redis Queue

```typescript
import { createRedisQueue } from "better-auth/queue";

backgroundTasks: {
  enabled: true,
  queue: createRedisQueue({
    url: process.env.REDIS_URL!,
    prefix: "auth-tasks:",
  }),
},
```

### Deferred Email Sending

```typescript
import { betterAuth } from "better-auth";

export const auth = betterAuth({
  backgroundTasks: { enabled: true },
  emailAndPassword: {
    enabled: true,
    sendVerificationEmail: async ({ user, url }) => {
      // This runs in background, not blocking the response
      await sendEmail({
        to: user.email,
        subject: "Verify your email",
        html: `<a href="${url}">Verify</a>`,
      });
    },
  },
});
```

---

## New OAuth Providers

### Patreon (v1.4.8)

```typescript
import { betterAuth } from "better-auth";

export const auth = betterAuth({
  socialProviders: {
    patreon: {
      clientId: process.env.PATREON_CLIENT_ID!,
      clientSecret: process.env.PATREON_CLIENT_SECRET!,
      // Scopes for patron data
      scope: ["identity", "identity[email]", "campaigns", "campaigns.members"],
    },
  },
});

// Client usage
await authClient.signIn.social({ provider: "patreon" });
```

### Vercel (v1.4.3)

```typescript
socialProviders: {
  vercel: {
    clientId: process.env.VERCEL_CLIENT_ID!,
    clientSecret: process.env.VERCEL_CLIENT_SECRET!,
    // Team-specific access
    teamId: process.env.VERCEL_TEAM_ID,
  },
},
```

### Kick (v1.4.6)

```typescript
socialProviders: {
  kick: {
    clientId: process.env.KICK_CLIENT_ID!,
    clientSecret: process.env.KICK_CLIENT_SECRET!,
    // Refresh token support
    refreshToken: true,
  },
},
```

---

## OAuth 2.1 Plugin (v1.4.8)

Standards-compliant OAuth 2.1 implementation.

### Configuration

```typescript
import { betterAuth } from "better-auth";
import { oauth21 } from "better-auth/plugins";

export const auth = betterAuth({
  plugins: [
    oauth21({
      // Enforce PKCE for all flows
      requirePKCE: true,
      // Enforce exact redirect URI matching
      strictRedirectUriMatching: true,
      // Disallow implicit grant
      disableImplicitGrant: true,
      // Short-lived access tokens
      accessTokenTTL: 60 * 15,  // 15 minutes
    }),
  ],
});
```

### OAuth 2.1 Compliance Features

- **PKCE Required**: All authorization flows must use PKCE
- **No Implicit Grant**: Token response type not allowed
- **Refresh Token Rotation**: New refresh token on each use
- **Exact Redirect URI Match**: No wildcard or partial matching
- **Short Access Tokens**: Encourages refresh token usage

---

## SAML/SSO Enhancements

### Clock Skew Tolerance (v1.4.7)

```typescript
import { sso } from "better-auth/plugins";

sso({
  saml: {
    // Allow 60 seconds of clock drift between SP and IdP
    clockSkewTolerance: 60,
  },
}),
```

### InResponseTo Validation (v1.4.7)

```typescript
sso({
  saml: {
    // Validate SAML response matches request
    validateInResponseTo: true,
    // Store pending requests (required for validation)
    requestStore: "redis",  // or "memory"
  },
}),
```

### OIDC Discovery for SSO (v1.4.7)

```typescript
sso({
  // Auto-configure from OIDC discovery document
  oidcDiscovery: true,
  providers: [
    {
      name: "Azure AD",
      // Discovery will fetch endpoints, keys, etc.
      discoveryUrl: "https://login.microsoftonline.com/{tenant}/.well-known/openid-configuration",
    },
  ],
}),
```

### Crypto Algorithm Validation (v1.4.8)

```typescript
sso({
  saml: {
    // Allowed signature algorithms
    signatureAlgorithms: ["sha256", "sha384", "sha512"],
    // Reject weak algorithms
    rejectWeakAlgorithms: true,  // Rejects sha1
  },
}),
```

---

## CLI Tool (v1.4.4)

better-auth now includes a CLI for project management.

### Installation

```bash
# Global installation
npm install -g better-auth

# Or use npx
npx better-auth
```

### Commands

```bash
# Initialize better-auth in project
better-auth init

# Generate auth schema
better-auth generate

# Check configuration
better-auth check

# Start development server
better-auth dev
```

### Directory Detection (v1.4.2)

The CLI automatically detects `/auth` directory for Next.js App Router:

```bash
# Detects src/app/api/auth/[...all]/route.ts
better-auth check
# ✓ Auth route detected at /api/auth
```

---

## Admin Role Permissions (v1.4.7)

Granular permissions for admin operations.

### Configuration

```typescript
import { admin } from "better-auth/plugins";

admin({
  adminRoles: ["admin", "super-admin"],
  // Granular permissions by role
  permissions: {
    admin: {
      updateUser: ["name", "image"],  // Can only update name/image
      deleteUser: false,              // Cannot delete
      banUser: true,
    },
    "super-admin": {
      updateUser: true,   // Can update any field
      deleteUser: true,
      banUser: true,
    },
  },
}),
```

### Permission Check

```typescript
// Server-side
const canDelete = await auth.api.admin.hasPermission({
  userId: session.user.id,
  permission: "deleteUser",
});
```

---

## Domain Verification Helpers

### ctx.isTrustedDomain (v1.4.6)

```typescript
// In custom endpoints or middleware
app.get("/api/secure", async (c) => {
  const ctx = auth.getContext(c);

  // Check if request origin is trusted
  if (!ctx.isTrustedDomain(c.req.headers.get("origin"))) {
    return c.json({ error: "Untrusted origin" }, 403);
  }

  // Proceed with trusted request
});
```

### Automatic Provider Trust (v1.4.6)

```typescript
socialProviders: {
  google: {
    clientId: "...",
    clientSecret: "...",
    // Trust verified domains from Google
    trustVerifiedDomains: true,
  },
},
```

---

## Verification Table Extensions (v1.4.8)

Add custom fields to verification records.

### Configuration

```typescript
betterAuth({
  // Add custom fields to verification table
  additionalFields: {
    verification: {
      // Custom field for tracking verification source
      source: {
        type: "string",
        required: false,
      },
      // Custom field for IP tracking
      requestIp: {
        type: "string",
        required: false,
      },
    },
  },
});
```

### Usage

```typescript
// When creating verification (internal)
await auth.api.createVerification({
  identifier: email,
  value: token,
  expiresAt: new Date(Date.now() + 3600000),
  // Custom fields
  source: "password-reset",
  requestIp: req.ip,
});
```

---

## GitHub PKCE Support (v1.4.2)

```typescript
socialProviders: {
  github: {
    clientId: process.env.GITHUB_CLIENT_ID!,
    clientSecret: process.env.GITHUB_CLIENT_SECRET!,
    // Enable PKCE for enhanced security
    pkce: true,
  },
},
```

---

## Trusted Proxy Headers (v1.4.3)

Properly infer base URL behind reverse proxies.

```typescript
betterAuth({
  // Trust X-Forwarded-* headers from proxy
  trustedProxyHeaders: true,
  // Or specify trusted proxies
  trustedProxies: ["10.0.0.0/8", "172.16.0.0/12"],
});
```

---

## Last Login Method (v1.4.6)

Track which method users last used to sign in.

### Server Configuration

```typescript
import { lastLoginMethod } from "better-auth/plugins";

betterAuth({
  plugins: [lastLoginMethod()],
});
```

### Client Usage (Expo-Optimized)

```typescript
import { lastLoginMethodClient } from "better-auth/client/plugins";

const authClient = createAuthClient({
  plugins: [lastLoginMethodClient()],
});

// Access in session
const { data: session } = useSession();
console.log(session?.user.lastLoginMethod);
// "email" | "google" | "github" | "passkey" | etc.
```

---

## Migration from v1.4.3

### Breaking Changes

None - all v1.4.x releases are backward compatible.

### Recommended Updates

1. **Enable Background Tasks** for email-heavy apps:
   ```typescript
   backgroundTasks: { enabled: true }
   ```

2. **Add PKCE to GitHub** if not using:
   ```typescript
   github: { pkce: true }
   ```

3. **Configure Clock Skew** for SAML if timing issues:
   ```typescript
   saml: { clockSkewTolerance: 60 }
   ```

4. **Use CLI** for configuration validation:
   ```bash
   npx better-auth check
   ```

---

## Cloudflare Workers Compatibility

All v1.4.x features are compatible with Cloudflare Workers:

- **Background Tasks**: Use in-memory queue (Redis not available on Workers)
- **OAuth Providers**: All new providers work on edge
- **SAML/SSO**: Full support with clock skew handling
- **CLI**: For local development, not edge runtime

### Workers-Specific Configuration

```typescript
// wrangler.jsonc
{
  "compatibility_flags": ["nodejs_compat"]
}
```

```typescript
// For background tasks on Workers
backgroundTasks: {
  enabled: true,
  queue: "memory",  // Only option for Workers
  // Consider using Cloudflare Queues for production
},
```

---

## Official Changelog

Full changelog: https://www.better-auth.com/changelogs
