# Template Structures

This document provides complete templates for all planning documents that can be generated by the project-planning skill.

---

## IMPLEMENTATION_PHASES.md Template

```markdown
# Implementation Phases: [Project Name]

**Project Type**: [Web App / Dashboard / API / etc]
**Stack**: Cloudflare Workers + Vite + React + D1
**Estimated Total**: [X hours] (~[Y minutes] human time)

---

## Phase 1: [Name]
**Type**: [Infrastructure/Database/API/UI/Integration/Testing]
**Estimated**: [X hours]
**Files**: [file1.ts, file2.tsx, ...]

**Tasks**:
- [ ] Task 1
- [ ] Task 2
- [ ] Task 3
- [ ] Test basic functionality

**Verification Criteria**:
- [ ] Specific test 1
- [ ] Specific test 2
- [ ] Specific test 3

**Exit Criteria**: [Clear definition of when this phase is complete]

---

## Phase 2: [Name]
[... repeat structure ...]

---

## Notes

**Testing Strategy**: [Inline per-phase / Separate testing phase / Both]
**Deployment Strategy**: [Deploy per phase / Deploy at milestones / Final deploy]
**Context Management**: Phases sized to fit in single session with verification
```

---

## DATABASE_SCHEMA.md Template

```markdown
# Database Schema: [Project Name]

**Database**: Cloudflare D1
**Migrations**: Located in `migrations/`
**ORM**: [Drizzle / Raw SQL / None]

---

## Tables

### `users`
**Purpose**: User accounts and authentication

| Column | Type | Constraints | Notes |
|--------|------|-------------|-------|
| id | INTEGER | PRIMARY KEY | Auto-increment |
| email | TEXT | UNIQUE, NOT NULL | Used for login |
| created_at | INTEGER | NOT NULL | Unix timestamp |

**Indexes**:
- `idx_users_email` on `email` (for login lookups)

**Relationships**:
- One-to-many with `tasks`

---

### `tasks`
[... repeat structure ...]

---

## Migrations

### Migration 1: Initial Schema
**File**: `migrations/0001_initial.sql`
**Creates**: users, tasks tables

### Migration 2: Add Tags
**File**: `migrations/0002_tags.sql`
**Creates**: tags, task_tags tables

---

## Seed Data

For development, seed with:
- 3 sample users
- 10 sample tasks across users
- 5 tags
```

---

## API_ENDPOINTS.md Template

```markdown
# API Endpoints: [Project Name]

**Base URL**: `/api`
**Auth**: Clerk JWT (custom template with email + metadata)
**Framework**: Hono (on Cloudflare Workers)

---

## Authentication

### POST /api/auth/verify
**Purpose**: Verify JWT token
**Auth**: None (public)
**Request**:
\`\`\`json
{
  "token": "string"
}
\`\`\`
**Responses**:
- 200: Token valid → `{ "valid": true, "email": "user@example.com" }`
- 401: Token invalid → `{ "error": "Invalid token" }`

---

## Users

### GET /api/users/me
**Purpose**: Get current user profile
**Auth**: Required (JWT)
**Responses**:
- 200: `{ "id": 1, "email": "user@example.com", "created_at": 1234567890 }`
- 401: Not authenticated

[... repeat for all endpoints ...]

---

## Error Handling

All endpoints return errors in this format:
\`\`\`json
{
  "error": "Human-readable message",
  "code": "ERROR_CODE",
  "details": {} // optional
}
\`\`\`

**Standard Codes**:
- 400: Bad request (validation failed)
- 401: Unauthorized (not logged in / invalid token)
- 403: Forbidden (insufficient permissions)
- 404: Not found
- 500: Internal server error
```

---

## ARCHITECTURE.md Template

```markdown
# Architecture: [Project Name]

**Deployment**: Cloudflare Workers
**Frontend**: Vite + React (served as static assets)
**Backend**: Worker handles API routes

---

## System Overview

\`\`\`
┌─────────────────┐
│   Browser       │
└────────┬────────┘
         │
         ↓ HTTPS
┌─────────────────────────────────────┐
│  Cloudflare Worker                  │
│  ┌──────────────┐  ┌──────────────┐│
│  │ Static Assets│  │  API Routes  ││
│  │ (Vite build) │  │    (Hono)    ││
│  └──────────────┘  └───────┬──────┘│
└─────────────────────────────┼───────┘
                              │
            ┌─────────────────┼─────────────────┐
            ↓                 ↓                 ↓
      ┌──────────┐      ┌──────────┐    ┌──────────┐
      │    D1    │      │    R2    │    │  Clerk   │
      │ (Database)│     │(Storage) │    │  (Auth)  │
      └──────────┘      └──────────┘    └──────────┘
\`\`\`

---

## Data Flow

### User Authentication
1. User submits login form
2. Frontend sends credentials to Clerk
3. Clerk returns JWT
4. Frontend includes JWT in API requests
5. Worker middleware verifies JWT
6. Protected routes accessible

### Task Creation
1. User submits task form
2. Frontend validates with Zod
3. POST /api/tasks with validated data
4. Worker validates again server-side
5. Insert into D1 database
6. Return created task
7. Frontend updates UI via TanStack Query

[... more flows as needed ...]

---

## Service Boundaries

**Frontend Responsibilities**:
- User interaction
- Client-side validation
- Optimistic updates
- State management (TanStack Query + Zustand)

**Worker Responsibilities**:
- Request routing
- Authentication/authorization
- Server-side validation
- Business logic
- Database operations
- Third-party API calls

**Cloudflare Services**:
- D1: Persistent relational data
- R2: File storage (images, documents)
- KV: Configuration, feature flags, cache

---

## Security

**Authentication**: Clerk JWT with custom claims
**Authorization**: Middleware checks user ownership before mutations
**Input Validation**: Zod schemas on client AND server
**CORS**: Restricted to production domain
**Secrets**: Environment variables in wrangler.jsonc (not committed)
```

---

## UI_COMPONENTS.md Template

```markdown
# UI Components: [Project Name]

**Framework**: React + TypeScript
**Styling**: Tailwind v4 + shadcn/ui
**State**: TanStack Query (server) + Zustand (client)
**Forms**: React Hook Form + Zod

---

## Component Hierarchy

\`\`\`
App
├── Layout
│   ├── Header
│   │   ├── Logo
│   │   ├── Navigation
│   │   └── UserMenu
│   ├── Sidebar (optional)
│   └── Footer
├── Pages
│   ├── HomePage
│   ├── TasksPage
│   │   ├── TaskList
│   │   │   └── TaskCard
│   │   ├── TaskDialog (create/edit)
│   │   └── TaskFilters
│   └── ProfilePage
│       └── ProfileForm
└── Shared Components
    ├── Button
    ├── Dialog
    ├── Form
    ├── Input
    └── Select
\`\`\`

---

## Component Details

### TaskList Component
**Purpose**: Display list of tasks with filtering and sorting
**Props**:
- `userId`: string (current user ID)
- `onTaskClick`: (task: Task) => void

**State**:
- Uses TanStack Query to fetch tasks
- Local filter state (completed, tags)

**Children**:
- TaskCard components for each task

---

### TaskDialog Component
**Purpose**: Create or edit a task
**Props**:
- `task`: Task | null (null for create, Task for edit)
- `onClose`: () => void
- `onSave`: (task: Task) => Promise<void>

**Form**:
- React Hook Form + Zod validation
- Fields: title (required), description, tags

**State**:
- Form state managed by React Hook Form
- Submission state (loading, error)

[... repeat for all major components ...]

---

## State Management

**Server State** (TanStack Query):
- Tasks list
- User profile
- API mutations (create, update, delete)

**Client State** (Zustand):
- UI state (dialog open/closed, selected filters)
- Theme preferences
- Sidebar collapsed state

---

## Shared Component Library

Use shadcn/ui components:
- Button
- Dialog
- Form
- Input
- Select
- Checkbox
- Label
- Card

Customize via Tailwind config for project-specific design.
```

---

## TESTING.md Template

```markdown
# Testing Strategy: [Project Name]

**Framework**: Vitest (unit) + Playwright (E2E)
**Coverage Goal**: 80% for critical paths
**CI/CD**: Run tests on every PR

---

## Unit Tests

**Scope**: Business logic, utilities, API route handlers

**Location**: `src/**/*.test.ts`

**Pattern**:
\`\`\`typescript
import { describe, it, expect } from 'vitest';
import { myFunction } from './myModule';

describe('myFunction', () => {
  it('should return expected result for valid input', () => {
    expect(myFunction('valid')).toBe('expected');
  });

  it('should throw error for invalid input', () => {
    expect(() => myFunction('invalid')).toThrow();
  });
});
\`\`\`

**Run**: `npm test`

---

## E2E Tests

**Scope**: Critical user flows (auth, CRUD operations)

**Location**: `tests/**/*.spec.ts`

**Pattern**:
\`\`\`typescript
import { test, expect } from '@playwright/test';

test('user can create a task', async ({ page }) => {
  await page.goto('/tasks');
  await page.click('button:has-text("New Task")');
  await page.fill('input[name="title"]', 'My Task');
  await page.click('button:has-text("Save")');
  await expect(page.locator('text=My Task')).toBeVisible();
});
\`\`\`

**Run**: `npm run test:e2e`

---

## Test Coverage

**Critical Paths** (must have tests):
- User authentication flow
- Task CRUD operations
- Form validation
- Error handling
- Authorization checks

**Nice to Have**:
- Edge cases
- Performance tests
- Load tests

---

## CI/CD Integration

\`\`\`yaml
name: Tests
on: [pull_request]
jobs:
  test:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      - run: npm install
      - run: npm test
      - run: npm run test:e2e
\`\`\`
```

---

## AGENTS_CONFIG.md Template

```markdown
# AI Agents Configuration: [Project Name]

**Provider**: [OpenAI / Anthropic / Cloudflare AI]
**Model**: [gpt-4 / claude-3-5-sonnet / llama-3]
**Use Case**: [Chat / Text generation / Data extraction]

---

## Agent Configuration

### Chat Agent
**Purpose**: Conversational AI for user support
**Model**: gpt-4-turbo
**System Prompt**:
\`\`\`
You are a helpful assistant for [Project Name]. You can:
- Answer questions about tasks
- Help create new tasks
- Provide usage tips

Be concise and friendly.
\`\`\`

**Context**:
- User's current tasks (last 10)
- User profile information

**Tools**:
- `create_task`: Create a new task
- `search_tasks`: Search user's tasks
- `get_task_details`: Get details for a specific task

---

## Token Management

**Max tokens per request**: 4000
**Context window**: 8000 tokens
**Strategy**: Summarize older messages when context exceeds 6000 tokens

---

## Error Handling

**Rate limits**: Retry with exponential backoff
**Service unavailable**: Show fallback message, queue request
**Invalid responses**: Log error, ask user to rephrase

---

## Cost Optimization

**Caching**: Cache common responses for 1 hour
**Model selection**: Use cheaper models for simple queries
**Streaming**: Enable streaming for better UX
```

---

## INTEGRATION.md Template

```markdown
# Third-Party Integrations: [Project Name]

**Purpose**: Document external service integrations, API keys, and webhooks

---

## Clerk Authentication

**Purpose**: User authentication and JWT management
**Docs**: https://clerk.com/docs

**Setup**:
1. Create Clerk application
2. Configure JWT template (include email, metadata)
3. Add `CLERK_SECRET_KEY` to wrangler.jsonc secrets
4. Install `@clerk/backend` package

**Usage**:
\`\`\`typescript
import { verifyToken } from '@clerk/backend';

const session = await verifyToken(token, {
  secretKey: env.CLERK_SECRET_KEY
});
\`\`\`

**Webhooks**:
- `user.created` → Create user record in D1
- `user.updated` → Update user record in D1

---

## Stripe Payments

**Purpose**: Payment processing
**Docs**: https://stripe.com/docs

**Setup**:
1. Create Stripe account
2. Add `STRIPE_SECRET_KEY` to wrangler.jsonc secrets
3. Install `stripe` package

**Webhooks**:
- `payment_intent.succeeded` → Mark order as paid
- `subscription.created` → Activate user subscription

**Testing**: Use Stripe test mode + test cards

---

## OpenAI API

**Purpose**: AI-powered features
**Docs**: https://platform.openai.com/docs

**Setup**:
1. Get API key from OpenAI
2. Add `OPENAI_API_KEY` to wrangler.jsonc secrets
3. Install `openai` package

**Rate Limits**: 60 requests/minute on tier 1
**Cost Management**: Cache responses, use cheaper models when possible

---

## Error Handling

**All integrations**:
- Implement retry logic with exponential backoff
- Log errors to monitoring service
- Show user-friendly error messages
- Have fallback behavior when service unavailable
```

---

## Usage Notes

All templates use:
- **Markdown formatting** for readability
- **Specific examples** rather than placeholders
- **Clear structure** for easy navigation
- **Default stack assumptions** (Cloudflare Workers, Vite, React)

Customize templates based on:
- User's actual stack choices
- Project complexity
- Feature requirements
- Team preferences
