---
name: vulnerability-scanning
description: Automated security scanning for dependencies, code, containers with Trivy, Snyk, npm audit. Use for CI/CD security gates, pre-deployment audits, compliance requirements, or encountering CVE detection, outdated packages, license compliance, SBOM generation errors.
keywords: Trivy, Snyk, npm-audit, OWASP, dependency-scanning, CVE, security-vulnerabilities, outdated-packages, license-compliance, SCA, SBOM, container-scanning, image-scanning, security-gates, CI-CD-security, pre-deployment-audit, supply-chain-security, vulnerability-detection, security-compliance, Docker-scan, Grype, static-analysis, dependency-check
license: MIT
---

# Vulnerability Scanning

Automate security vulnerability detection across code, dependencies, and containers.

## Dependency Scanning

```bash
# npm audit
npm audit --audit-level=high

# Snyk
snyk test --severity-threshold=high

# Safety (Python)
safety check --full-report
```

## Container Scanning (Trivy)

```bash
# Scan container image
trivy image myapp:latest --severity HIGH,CRITICAL

# Scan filesystem
trivy fs --scanners vuln,secret .
```

## GitHub Actions Integration

```yaml
name: Security Scan

on: [push, pull_request]

jobs:
  security:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Run Trivy vulnerability scanner
        uses: aquasecurity/trivy-action@master
        with:
          scan-type: 'fs'
          severity: 'CRITICAL,HIGH'
          exit-code: '1'

      - name: Run Snyk
        uses: snyk/actions/node@master
        env:
          SNYK_TOKEN: ${{ secrets.SNYK_TOKEN }}
        with:
          args: --severity-threshold=high

      - name: npm audit
        run: npm audit --audit-level=high
```

## Code Analysis (Bandit for Python)

```bash
bandit -r src/ -ll -ii
```

## Node.js Scanner

```javascript
const { execSync } = require('child_process');

function runSecurityScan() {
  const results = {
    npm: JSON.parse(execSync('npm audit --json').toString()),
    trivy: JSON.parse(execSync('trivy fs --format json .').toString())
  };

  const critical = results.npm.metadata?.vulnerabilities?.critical || 0;
  if (critical > 0) {
    console.error(`Found ${critical} critical vulnerabilities`);
    process.exit(1);
  }
}
```

## Best Practices

- Integrate scanning in CI/CD pipeline
- Fail builds on high/critical findings
- Scan dependencies and containers
- Track vulnerabilities over time
- Document accepted false positives

## Tools

- Trivy (containers, filesystem)
- Snyk (dependencies, code)
- npm audit / yarn audit
- Bandit (Python)
- OWASP Dependency-Check
