# ReportLab API Quick Reference

Quick reference for commonly used ReportLab functions.

## Canvas API (Low-Level)

### Creating a Canvas

```python
from reportlab.pdfgen import canvas
from reportlab.lib.pagesizes import letter, A4
from reportlab.lib.units import inch, cm

c = canvas.Canvas("output.pdf", pagesize=letter)
width, height = letter

# Save when done
c.save()
```

### Text Drawing

```python
# Simple text
c.drawString(x, y, "Text")
c.drawRightString(x, y, "Right-aligned")
c.drawCentredString(x, y, "Centered")

# Font and size
c.setFont("Helvetica", 12)
c.setFont("Helvetica-Bold", 16)
c.setFont("Times-Roman", 14)

# Text color
from reportlab.lib import colors
c.setFillColor(colors.red)
c.setFillColor(colors.HexColor('#FF5733'))

# Text width (for alignment)
width = c.stringWidth("Text", "Helvetica", 12)

# Multi-line text
textobject = c.beginText(x, y)
textobject.setFont("Helvetica", 12)
textobject.textLine("Line 1")
textobject.textLine("Line 2")
c.drawText(textobject)
```

### Shapes

```python
# Line
c.line(x1, y1, x2, y2)

# Rectangle
c.rect(x, y, width, height, stroke=1, fill=0)

# Circle
c.circle(x_center, y_center, radius, stroke=1, fill=0)

# Ellipse
c.ellipse(x1, y1, x2, y2, stroke=1, fill=0)

# Rounded rectangle
c.roundRect(x, y, width, height, radius, stroke=1, fill=0)

# Colors
c.setStrokeColor(colors.black)
c.setFillColor(colors.blue)
c.setLineWidth(2)
```

### Images

```python
# Draw image
c.drawImage("image.png", x, y, width=w, height=h)

# With preserved aspect ratio
c.drawImage("photo.jpg", x, y, width=w, preserveAspectRatio=True)

# Inline image (from PIL)
from PIL import Image
img = Image.open("photo.jpg")
c.drawInlineImage(img, x, y, width=w, height=h)
```

### Page Operations

```python
# New page
c.showPage()

# Set page size
c.setPageSize(letter)
```

## Platypus API (High-Level)

### Document Setup

```python
from reportlab.platypus import SimpleDocTemplate
from reportlab.lib.pagesizes import letter
from reportlab.lib.units import inch

doc = SimpleDocTemplate(
    "output.pdf",
    pagesize=letter,
    leftMargin=inch,
    rightMargin=inch,
    topMargin=inch,
    bottomMargin=inch
)

story = []  # List of flowables
# Add content to story...
doc.build(story)
```

### Paragraphs

```python
from reportlab.platypus import Paragraph
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.enums import TA_LEFT, TA_CENTER, TA_RIGHT, TA_JUSTIFY

styles = getSampleStyleSheet()

# Built-in styles
p = Paragraph("Text", styles['Normal'])
p = Paragraph("Title", styles['Title'])
p = Paragraph("Heading 1", styles['Heading1'])

# Custom style
custom = ParagraphStyle(
    'CustomStyle',
    parent=styles['Normal'],
    fontSize=14,
    textColor=colors.blue,
    alignment=TA_CENTER,
    spaceAfter=10,
    leading=16,  # Line height
    leftIndent=20,
    rightIndent=20
)
p = Paragraph("Custom text", custom)
```

### Tables

```python
from reportlab.platypus import Table, TableStyle
from reportlab.lib import colors

data = [
    ['Header 1', 'Header 2', 'Header 3'],
    ['Row 1, Col 1', 'Row 1, Col 2', 'Row 1, Col 3'],
    ['Row 2, Col 1', 'Row 2, Col 2', 'Row 2, Col 3'],
]

table = Table(data, colWidths=[2*inch, 2*inch, 2*inch])

table.setStyle(TableStyle([
    # Cell range: (start_col, start_row), (end_col, end_row)
    # Use -1 for last row/column

    # Background
    ('BACKGROUND', (0, 0), (-1, 0), colors.grey),

    # Text color
    ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),

    # Alignment
    ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
    ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),

    # Font
    ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
    ('FONTSIZE', (0, 0), (-1, 0), 12),

    # Padding
    ('BOTTOMPADDING', (0, 0), (-1, 0), 12),
    ('TOPPADDING', (0, 1), (-1, -1), 6),

    # Grid
    ('GRID', (0, 0), (-1, -1), 1, colors.black),
    ('BOX', (0, 0), (-1, -1), 2, colors.black),
    ('LINEBEFORE', (1, 0), (1, -1), 1, colors.grey),

    # Spans
    # ('SPAN', (0, 0), (1, 0)),  # Merge cells
]))
```

### Spacers

```python
from reportlab.platypus import Spacer
from reportlab.lib.units import inch

# Add vertical space
story.append(Spacer(1, 0.5*inch))
```

### Page Breaks

```python
from reportlab.platypus import PageBreak

story.append(PageBreak())
```

### Images in Platypus

```python
from reportlab.platypus import Image
from reportlab.lib.units import inch

img = Image("photo.jpg", width=4*inch, height=3*inch)
story.append(img)

# With preserved aspect ratio
img = Image("photo.jpg", width=4*inch, height=3*inch, kind='proportional')
```

### Charts

```python
from reportlab.graphics.shapes import Drawing
from reportlab.graphics.charts.barcharts import VerticalBarChart
from reportlab.graphics.charts.piecharts import Pie
from reportlab.graphics.charts.linecharts import HorizontalLineChart

# Bar chart
drawing = Drawing(400, 200)
chart = VerticalBarChart()
chart.x = 50
chart.y = 50
chart.height = 125
chart.width = 300
chart.data = [[100, 150, 200, 175]]
chart.categoryAxis.categoryNames = ['Q1', 'Q2', 'Q3', 'Q4']
chart.bars[0].fillColor = colors.blue
drawing.add(chart)
story.append(drawing)

# Pie chart
drawing = Drawing(400, 200)
pie = Pie()
pie.x = 150
pie.y = 50
pie.width = 100
pie.height = 100
pie.data = [30, 25, 20, 15, 10]
pie.labels = ['A', 'B', 'C', 'D', 'E']
drawing.add(pie)
story.append(drawing)
```

## Common Utilities

### Page Sizes

```python
from reportlab.lib.pagesizes import letter, A4, legal, A3, A5
from reportlab.lib.pagesizes import landscape, portrait

# Standard sizes
letter  # 8.5" x 11"
A4      # 210mm x 297mm
legal   # 8.5" x 14"

# Orientation
landscape(letter)
portrait(A4)

# Custom
(width, height) = (8.5*inch, 11*inch)
```

### Units

```python
from reportlab.lib.units import inch, cm, mm, pica

# Conversions
1*inch  # 72 points
1*cm    # 28.35 points
1*mm    # 2.83 points
1*pica  # 12 points
```

### Colors

```python
from reportlab.lib import colors

# Named colors
colors.red, colors.green, colors.blue
colors.black, colors.white, colors.grey

# RGB
colors.Color(1, 0, 0)  # Red (values 0-1)

# Hex
colors.HexColor('#FF5733')

# CMYK
colors.CMYKColor(0, 1, 1, 0)  # Red
```

### Fonts

```python
# Standard fonts (always available)
"Helvetica"
"Helvetica-Bold"
"Helvetica-Oblique"
"Helvetica-BoldOblique"
"Times-Roman"
"Times-Bold"
"Times-Italic"
"Times-BoldItalic"
"Courier"
"Courier-Bold"
"Courier-Oblique"
"Courier-BoldOblique"

# Register TrueType font
from reportlab.pdfbase import pdfmetrics
from reportlab.pdfbase.ttfonts import TTFont

pdfmetrics.registerFont(TTFont('CustomFont', 'font.ttf'))
```

## Header/Footer Template

```python
def add_header_footer(canvas, doc):
    canvas.saveState()

    # Header
    canvas.setFont('Helvetica-Bold', 10)
    canvas.drawString(inch, letter[1] - 0.5*inch, "Header Text")

    # Footer
    canvas.setFont('Helvetica', 9)
    canvas.drawString(inch, 0.5*inch, f"Page {doc.page}")

    canvas.restoreState()

# Use with document
doc.build(story, onFirstPage=add_header_footer, onLaterPages=add_header_footer)
```

## Useful Patterns

### Coordinate System
```python
# (0, 0) is bottom-left corner
# y increases upward
# x increases rightward

width, height = letter
top_y = height - margin
bottom_y = margin
```

### Safe Drawing Area
```python
from reportlab.lib.pagesizes import letter

width, height = letter
margin = 0.75*inch
x_min = margin
x_max = width - margin
y_min = margin
y_max = height - margin
```

## Additional Resources

- Official docs: https://docs.reportlab.com/
- User guide PDF: https://www.reportlab.com/docs/reportlab-userguide.pdf
- PyPI: https://pypi.org/project/reportlab/
