## Common Patterns

### Pattern 1: Invoice Generator

```python
from reportlab.lib.pagesizes import letter
from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from reportlab.lib import colors
from reportlab.lib.enums import TA_RIGHT
from datetime import datetime

def create_invoice(invoice_number, customer_name, items, output_file="invoice.pdf"):
    doc = SimpleDocTemplate(output_file, pagesize=letter)
    story = []
    styles = getSampleStyleSheet()

    # Header
    story.append(Paragraph("INVOICE", styles['Title']))
    story.append(Spacer(1, 0.2*inch))

    # Invoice details
    invoice_info = [
        ['Invoice Number:', invoice_number],
        ['Date:', datetime.now().strftime('%Y-%m-%d')],
        ['Customer:', customer_name],
    ]
    info_table = Table(invoice_info, colWidths=[2*inch, 3*inch])
    info_table.setStyle(TableStyle([
        ('ALIGN', (0, 0), (0, -1), 'RIGHT'),
        ('FONTNAME', (0, 0), (0, -1), 'Helvetica-Bold'),
    ]))
    story.append(info_table)
    story.append(Spacer(1, 0.3*inch))

    # Items table
    table_data = [['Description', 'Quantity', 'Unit Price', 'Total']]
    subtotal = 0

    for item in items:
        total = item['quantity'] * item['price']
        subtotal += total
        table_data.append([
            item['description'],
            str(item['quantity']),
            f"${item['price']:.2f}",
            f"${total:.2f}"
        ])

    # Add totals
    tax = subtotal * 0.1  # 10% tax
    total = subtotal + tax

    table_data.append(['', '', 'Subtotal:', f"${subtotal:.2f}"])
    table_data.append(['', '', 'Tax (10%):', f"${tax:.2f}"])
    table_data.append(['', '', 'Total:', f"${total:.2f}"])

    items_table = Table(table_data, colWidths=[3*inch, 1*inch, 1.5*inch, 1.5*inch])
    items_table.setStyle(TableStyle([
        # Header
        ('BACKGROUND', (0, 0), (-1, 0), colors.grey),
        ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, 0), 'CENTER'),
        ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),

        # Data
        ('ALIGN', (1, 1), (-1, -1), 'RIGHT'),
        ('GRID', (0, 0), (-1, -4), 1, colors.black),

        # Totals
        ('FONTNAME', (2, -3), (-1, -1), 'Helvetica-Bold'),
        ('LINEABOVE', (2, -3), (-1, -3), 1, colors.black),
        ('LINEABOVE', (2, -1), (-1, -1), 2, colors.black),
    ]))

    story.append(items_table)
    doc.build(story)

# Usage
items = [
    {'description': 'Widget A', 'quantity': 5, 'price': 29.99},
    {'description': 'Widget B', 'quantity': 2, 'price': 49.99},
    {'description': 'Service Fee', 'quantity': 1, 'price': 100.00},
]
create_invoice("INV-2025-001", "Acme Corporation", items)
```

### Pattern 2: Certificate Generator

```python
from reportlab.pdfgen import canvas
from reportlab.lib.pagesizes import landscape, A4
from reportlab.lib.units import inch
from reportlab.lib import colors
from reportlab.pdfbase import pdfmetrics
from reportlab.pdfbase.ttfonts import TTFont

def create_certificate(name, course, date, output_file="certificate.pdf"):
    c = canvas.Canvas(output_file, pagesize=landscape(A4))
    width, height = landscape(A4)

    # Border
    c.setStrokeColor(colors.gold)
    c.setLineWidth(5)
    c.rect(0.5*inch, 0.5*inch, width - inch, height - inch)

    # Decorative inner border
    c.setStrokeColor(colors.darkgoldenrod)
    c.setLineWidth(2)
    c.rect(0.75*inch, 0.75*inch, width - 1.5*inch, height - 1.5*inch)

    # Title
    c.setFont("Helvetica-Bold", 48)
    c.setFillColor(colors.darkblue)
    c.drawCentredString(width/2, height - 2*inch, "Certificate of Completion")

    # Body text
    c.setFont("Helvetica", 20)
    c.setFillColor(colors.black)
    c.drawCentredString(width/2, height - 3*inch, "This is to certify that")

    # Name (large and prominent)
    c.setFont("Helvetica-Bold", 36)
    c.setFillColor(colors.darkblue)
    c.drawCentredString(width/2, height - 4*inch, name)

    # Course info
    c.setFont("Helvetica", 18)
    c.setFillColor(colors.black)
    c.drawCentredString(width/2, height - 5*inch, "has successfully completed the course")

    c.setFont("Helvetica-Bold", 24)
    c.drawCentredString(width/2, height - 5.75*inch, course)

    # Date
    c.setFont("Helvetica", 14)
    c.drawCentredString(width/2, height - 6.5*inch, f"Date: {date}")

    c.save()

# Usage
create_certificate("John Doe", "Advanced Python Programming", "2025-10-27")
```

### Pattern 3: Multi-Page Report

```python
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, PageBreak, Table, TableStyle
from reportlab.lib.pagesizes import letter
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from reportlab.lib import colors

def header_footer(canvas, doc):
    """Add header and footer to each page"""
    canvas.saveState()

    # Header
    canvas.setFont('Helvetica-Bold', 10)
    canvas.drawString(inch, letter[1] - 0.5*inch, "Company Report - Confidential")

    # Footer
    canvas.setFont('Helvetica', 9)
    canvas.drawString(inch, 0.5*inch, f"Page {doc.page}")
    canvas.drawRightString(letter[0] - inch, 0.5*inch, "© 2025 Company Name")

    canvas.restoreState()

def create_report(output_file="report.pdf"):
    doc = SimpleDocTemplate(output_file, pagesize=letter)
    story = []
    styles = getSampleStyleSheet()

    # Cover page
    story.append(Spacer(1, 2*inch))
    story.append(Paragraph("Annual Report 2025", styles['Title']))
    story.append(Spacer(1, 0.5*inch))
    story.append(Paragraph("Company Performance Analysis", styles['Heading1']))
    story.append(PageBreak())

    # Executive Summary
    story.append(Paragraph("Executive Summary", styles['Heading1']))
    story.append(Spacer(1, 0.2*inch))
    summary_text = "This report provides a comprehensive analysis of company performance..." * 5
    story.append(Paragraph(summary_text, styles['Normal']))
    story.append(Spacer(1, 0.3*inch))

    # Financial Data
    story.append(Paragraph("Financial Performance", styles['Heading1']))
    story.append(Spacer(1, 0.2*inch))

    financial_data = [
        ['Metric', 'Q1', 'Q2', 'Q3', 'Q4'],
        ['Revenue', '$2.5M', '$2.8M', '$3.1M', '$3.5M'],
        ['Expenses', '$1.8M', '$1.9M', '$2.0M', '$2.1M'],
        ['Profit', '$0.7M', '$0.9M', '$1.1M', '$1.4M'],
    ]

    table = Table(financial_data)
    table.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (-1, 0), colors.grey),
        ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
        ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
        ('GRID', (0, 0), (-1, -1), 1, colors.black),
        ('BACKGROUND', (0, 1), (-1, -1), colors.beige),
    ]))
    story.append(table)

    # Build with header/footer
    doc.build(story, onFirstPage=header_footer, onLaterPages=header_footer)

create_report()
```

### Pattern 4: Data-Driven PDF from DataFrame

```python
from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph
from reportlab.lib.pagesizes import letter
from reportlab.lib import colors
from reportlab.lib.styles import getSampleStyleSheet
import pandas as pd

def dataframe_to_pdf(df, title, output_file="data_report.pdf"):
    doc = SimpleDocTemplate(output_file, pagesize=letter)
    story = []
    styles = getSampleStyleSheet()

    # Title
    story.append(Paragraph(title, styles['Title']))

    # Convert DataFrame to list
    data = [df.columns.tolist()] + df.values.tolist()

    # Create table
    table = Table(data)
    table.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (-1, 0), colors.grey),
        ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
        ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
        ('FONTSIZE', (0, 0), (-1, 0), 12),
        ('BOTTOMPADDING', (0, 0), (-1, 0), 12),
        ('BACKGROUND', (0, 1), (-1, -1), colors.beige),
        ('GRID', (0, 0), (-1, -1), 1, colors.black),
    ]))

    story.append(table)
    doc.build(story)

# Usage
df = pd.DataFrame({
    'Product': ['A', 'B', 'C'],
    'Sales': [100, 150, 200],
    'Revenue': [1000, 1500, 2000]
})
dataframe_to_pdf(df, "Sales Report", "sales_report.pdf")
```
