# 代码审查反馈指南

## 反馈原则

### 建设性与尊重
代码审查的目标是提升代码质量和团队学习，而非批评个人。反馈应当：
- **专注于代码，而非编写者**：说"这个函数可以更简洁"而不是"你写得太复杂了"
- **保持专业和尊重**：即使发现严重问题，也要以建设性的方式提出
- **提供具体建议**：不要只说"这不好"，要说明为什么不好以及如何改进
- **认可好的实践**：发现优秀代码时也要给予肯定

### 清晰与具体
- **明确问题所在**：指出具体的行号和代码段
- **说明影响**：解释问题可能导致的后果（安全、性能、可维护性等）
- **提供上下文**：如果引用外部资源或标准，提供链接
- **举例说明**：必要时提供代码示例展示建议的改进方式

### 优先级分级
使用标签或关键词区分问题的严重程度：
- **🔴 必须修复（Blocker）**：安全漏洞、严重bug、违反核心规范
- **🟡 强烈建议（Major）**：性能问题、设计缺陷、可维护性问题
- **🟢 建议（Minor）**：代码风格、命名优化、轻微改进
- **💡 可选（Nit）**：个人偏好、非强制性建议

---

## 反馈模式

### 提问式反馈
鼓励对话和思考：
- "这里为什么选择使用X而不是Y？"
- "这个逻辑能否处理边界情况Z？"
- "我们是否考虑过使用设计模式A来解决这个问题？"

### 建议式反馈
提供具体的改进建议：
```
建议：考虑将这个200行的函数拆分为多个职责单一的小函数，以提高可读性和可测试性。

例如：
- extractUserData()
- validateUserInput()
- processUserRegistration()
```

### 解释式反馈
说明问题的原因和影响：
```
这里直接使用用户输入拼接SQL查询可能导致SQL注入攻击。
建议使用参数化查询或ORM来防止此类安全漏洞。

示例（使用参数化查询）：
cursor.execute("SELECT * FROM users WHERE id = ?", (user_id,))
```

### 赞赏式反馈
认可优秀的实践：
- "👍 这个错误处理逻辑写得很清晰！"
- "✨ 很好的抽象，这个设计使代码更易于扩展"
- "💯 测试覆盖很完整，包含了边界情况"

---

## 常见反馈场景

### 安全问题
```
🔴 必须修复：这里使用了硬编码的密码。
影响：敏感信息泄露风险
建议：将密码移至环境变量或密钥管理服务中。

参考：https://owasp.org/www-project-top-ten/
```

### 性能问题
```
🟡 强烈建议：这个循环内的数据库查询会导致N+1问题。
影响：当数据量增大时，性能会显著下降
建议：使用JOIN或预加载来优化查询。

修改前：
for order in orders:
    customer = db.query("SELECT * FROM customers WHERE id = ?", order.customer_id)

修改后：
orders = db.query("SELECT * FROM orders JOIN customers ON orders.customer_id = customers.id")
```

### 代码质量
```
🟢 建议：这个函数名`doStuff`不够清晰。
影响：降低代码可读性
建议：根据函数实际功能重命名，如`processUserRegistration`或`validateEmailFormat`。
```

### 测试缺失
```
🟡 强烈建议：新增的功能缺少单元测试。
影响：未来修改可能引入回归bug
建议：至少添加以下测试用例：
- 正常输入的成功场景
- 边界值测试（空值、最大值、最小值）
- 异常情况处理
```

### 文档不足
```
🟢 建议：这个公共API函数缺少文档注释。
影响：其他开发者难以理解如何使用
建议：添加文档注释说明参数、返回值和可能的异常。

示例：
/**
 * 根据用户ID获取用户信息
 * @param userId - 用户的唯一标识符
 * @returns 用户对象，如果不存在则返回null
 * @throws DatabaseError - 数据库连接失败时抛出
 */
function getUserById(userId: string): User | null
```

---

## 反馈时机

### 审查过程中
- **及时反馈**：发现问题时立即留下评论，不要等到审查结束
- **分批处理**：如果发现大量问题，优先标注最关键的问题
- **持续对话**：作者回复后及时跟进讨论

### 审查结束时
提供总结性评论：
```
总体评价：✅ 代码功能实现正确，逻辑清晰。

主要问题：
1. [必须修复] 安全：API端点缺少认证检查（line 45）
2. [强烈建议] 性能：循环内的数据库查询需要优化（line 78-82）
3. [建议] 测试：建议增加边界情况的单元测试

优点：
- 错误处理做得很好
- 代码结构清晰
- 命名规范一致

下一步：修复必须项后我会再次审查，其他建议可在后续PR中改进。
```

---

## 避免的反馈方式

### ❌ 不要这样
- "这代码写得太烂了" - 过于主观和负面
- "为什么不用X？" - 没有解释理由
- "LGTM"（Looks Good To Me）- 没有实际审查就批准
- "按我说的做" - 过于强制，不鼓励讨论
- 只指出问题但不提供建议 - 不够建设性

### ✅ 应该这样
- "这里的逻辑可以简化。建议使用X模式，因为它更符合项目规范且更易维护"
- "考虑使用X的原因是：1) 性能更好 2) 代码更简洁 3) 社区推荐的最佳实践"
- "我仔细审查了这段代码，功能实现正确，但有几个小建议..."
- "我建议使用X，但如果你有其他考虑我们可以讨论"
- "这里存在Y问题。建议改为Z，示例代码如下：..."

---

## 响应作者的反馈

### 作者提出质疑时
- 保持开放心态，可能是你理解有误
- 礼貌地解释你的观点和理由
- 如果作者的解释合理，承认并撤回你的建议
- 如果意见分歧，寻求第三方意见或团队讨论

### 作者同意修改时
- 表示感谢和肯定
- 如果是重大修改，建议再次审查
- 标记评论为"已解决"以保持清晰

### 作者没有回应时
- 礼貌地提醒："嗨，关于X的建议有什么想法吗？"
- 如果是必须修复的问题，明确说明无法批准PR的原因
- 如果是次要建议，可以考虑在后续PR中处理

---

## 特殊场景处理

### 初级开发者的代码
- 更加耐心和详细地解释
- 提供学习资源和文档链接
- 鼓励提问和讨论
- 认可进步和亮点

### 紧急修复
- 专注于关键问题（安全、功能正确性）
- 次要改进可以在后续PR中处理
- 明确标注"技术债务"以便跟踪

### 大型PR
- 如果PR过大难以审查，建议拆分
- 先审查关键路径和核心逻辑
- 可以要求简要演示或解释设计思路

### 与代码作者意见不合
- 基于事实和标准，而非个人偏好
- 引用项目规范、行业最佳实践或技术文档
- 寻求团队共识或技术负责人的意见
- 如果是风格问题，遵循"代码一致性优先"原则

---

## 自动化辅助

建议使用工具辅助代码审查：
- **静态代码分析**：SonarQube、ESLint、Pylint等
- **安全扫描**：Snyk、OWASP Dependency-Check等
- **格式化工具**：Prettier、Black等
- **测试覆盖率**：Codecov、Coveralls等

这些工具可以自动发现常见问题，让审查者专注于逻辑和架构层面的问题。
