'use client'

import * as SignIn from '@clerk/elements/sign-in'
import { useState } from 'react'

/**
 * Custom Sign-In Component using Clerk Elements
 *
 * This example demonstrates:
 * - Complete control over sign-in flow
 * - Custom styling and branding
 * - Multiple authentication strategies (password, email code, OAuth)
 * - Error handling and validation
 * - Loading states
 */
export default function CustomSignInPage() {
  return (
    <div className="flex min-h-screen items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100 p-4">
      <div className="w-full max-w-md">
        <SignIn.Root>
          {/* Start step - email/username input */}
          <SignIn.Step name="start" className="bg-white rounded-2xl shadow-xl p-8">
            <div className="mb-8 text-center">
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Welcome back</h1>
              <p className="text-gray-600">Sign in to your account to continue</p>
            </div>

            {/* Email/Username input */}
            <div className="space-y-4">
              <div>
                <label htmlFor="identifier" className="block text-sm font-medium text-gray-700 mb-1">
                  Email or username
                </label>
                <SignIn.Input
                  type="text"
                  name="identifier"
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                  placeholder="Enter your email or username"
                />
              </div>

              {/* Continue button */}
              <SignIn.Action
                submit
                className="w-full bg-blue-600 text-white py-3 px-4 rounded-lg font-semibold hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Continue
              </SignIn.Action>
            </div>

            {/* Divider */}
            <div className="relative my-6">
              <div className="absolute inset-0 flex items-center">
                <div className="w-full border-t border-gray-300" />
              </div>
              <div className="relative flex justify-center text-sm">
                <span className="px-2 bg-white text-gray-500">Or continue with</span>
              </div>
            </div>

            {/* Social authentication buttons */}
            <div className="space-y-3">
              <SignIn.Strategy name="oauth_google">
                <button className="w-full flex items-center justify-center gap-3 px-4 py-3 border-2 border-gray-300 rounded-lg hover:border-gray-400 transition-colors">
                  <GoogleIcon />
                  <span className="font-medium text-gray-700">Continue with Google</span>
                </button>
              </SignIn.Strategy>

              <SignIn.Strategy name="oauth_github">
                <button className="w-full flex items-center justify-center gap-3 px-4 py-3 border-2 border-gray-300 rounded-lg hover:border-gray-400 transition-colors">
                  <GithubIcon />
                  <span className="font-medium text-gray-700">Continue with GitHub</span>
                </button>
              </SignIn.Strategy>
            </div>

            {/* Sign up link */}
            <p className="mt-6 text-center text-sm text-gray-600">
              Don't have an account?{' '}
              <a href="/sign-up" className="font-semibold text-blue-600 hover:text-blue-700">
                Sign up
              </a>
            </p>
          </SignIn.Step>

          {/* Verification step - password or email code */}
          <SignIn.Step name="verifications" className="bg-white rounded-2xl shadow-xl p-8">
            <div className="mb-8 text-center">
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Verify your identity</h1>
              <p className="text-gray-600">Choose how you'd like to sign in</p>
            </div>

            {/* Password strategy */}
            <SignIn.Strategy name="password">
              <div className="space-y-4">
                <div>
                  <label htmlFor="password" className="block text-sm font-medium text-gray-700 mb-1">
                    Password
                  </label>
                  <SignIn.Input
                    type="password"
                    name="password"
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                    placeholder="Enter your password"
                  />
                </div>

                <SignIn.Action
                  submit
                  className="w-full bg-blue-600 text-white py-3 px-4 rounded-lg font-semibold hover:bg-blue-700 transition-colors"
                >
                  Sign in
                </SignIn.Action>

                <SignIn.Action
                  navigate="forgot-password"
                  className="w-full text-center text-sm text-blue-600 hover:text-blue-700"
                >
                  Forgot password?
                </SignIn.Action>
              </div>
            </SignIn.Strategy>

            {/* Email code strategy */}
            <SignIn.Strategy name="email_code">
              <div className="space-y-4">
                <p className="text-sm text-gray-600">
                  We'll send a verification code to your email
                </p>

                <SignIn.Action
                  submit
                  className="w-full bg-blue-600 text-white py-3 px-4 rounded-lg font-semibold hover:bg-blue-700 transition-colors"
                >
                  Send code
                </SignIn.Action>

                <div>
                  <label htmlFor="code" className="block text-sm font-medium text-gray-700 mb-1">
                    Verification code
                  </label>
                  <SignIn.Input
                    type="text"
                    name="code"
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                    placeholder="Enter 6-digit code"
                  />
                </div>
              </div>
            </SignIn.Strategy>

            {/* Choose different strategy */}
            <SignIn.Action
              navigate="choose-strategy"
              className="mt-4 w-full text-center text-sm text-gray-600 hover:text-gray-900"
            >
              Use a different method
            </SignIn.Action>
          </SignIn.Step>

          {/* Choose strategy step */}
          <SignIn.Step name="choose-strategy" className="bg-white rounded-2xl shadow-xl p-8">
            <div className="mb-8 text-center">
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Choose verification method</h1>
              <p className="text-gray-600">Select how you'd like to verify your identity</p>
            </div>

            <div className="space-y-3">
              <SignIn.Strategy name="password">
                <button className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg hover:border-blue-500 transition-colors text-left">
                  <div className="font-medium text-gray-900">Password</div>
                  <div className="text-sm text-gray-600">Use your password to sign in</div>
                </button>
              </SignIn.Strategy>

              <SignIn.Strategy name="email_code">
                <button className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg hover:border-blue-500 transition-colors text-left">
                  <div className="font-medium text-gray-900">Email code</div>
                  <div className="text-sm text-gray-600">Get a code sent to your email</div>
                </button>
              </SignIn.Strategy>
            </div>
          </SignIn.Step>

          {/* Forgot password step */}
          <SignIn.Step name="forgot-password" className="bg-white rounded-2xl shadow-xl p-8">
            <div className="mb-8 text-center">
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Reset your password</h1>
              <p className="text-gray-600">We'll send a reset code to your email</p>
            </div>

            <div className="space-y-4">
              <SignIn.Action
                submit
                className="w-full bg-blue-600 text-white py-3 px-4 rounded-lg font-semibold hover:bg-blue-700 transition-colors"
              >
                Send reset code
              </SignIn.Action>

              <SignIn.Action
                navigate="start"
                className="w-full text-center text-sm text-gray-600 hover:text-gray-900"
              >
                Back to sign in
              </SignIn.Action>
            </div>
          </SignIn.Step>

          {/* Reset password step */}
          <SignIn.Step name="reset-password" className="bg-white rounded-2xl shadow-xl p-8">
            <div className="mb-8 text-center">
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Create new password</h1>
              <p className="text-gray-600">Enter your new password below</p>
            </div>

            <div className="space-y-4">
              <div>
                <label htmlFor="code" className="block text-sm font-medium text-gray-700 mb-1">
                  Reset code
                </label>
                <SignIn.Input
                  type="text"
                  name="code"
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter reset code"
                />
              </div>

              <div>
                <label htmlFor="password" className="block text-sm font-medium text-gray-700 mb-1">
                  New password
                </label>
                <SignIn.Input
                  type="password"
                  name="password"
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter new password"
                />
              </div>

              <SignIn.Action
                submit
                className="w-full bg-blue-600 text-white py-3 px-4 rounded-lg font-semibold hover:bg-blue-700 transition-colors"
              >
                Reset password
              </SignIn.Action>
            </div>
          </SignIn.Step>
        </SignIn.Root>
      </div>
    </div>
  )
}

// Icon components (simplified SVGs)
function GoogleIcon() {
  return (
    <svg className="w-5 h-5" viewBox="0 0 24 24">
      <path fill="#4285F4" d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"/>
      <path fill="#34A853" d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"/>
      <path fill="#FBBC05" d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"/>
      <path fill="#EA4335" d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"/>
    </svg>
  )
}

function GithubIcon() {
  return (
    <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
      <path fillRule="evenodd" d="M12 2C6.477 2 2 6.484 2 12.017c0 4.425 2.865 8.18 6.839 9.504.5.092.682-.217.682-.483 0-.237-.008-.868-.013-1.703-2.782.605-3.369-1.343-3.369-1.343-.454-1.158-1.11-1.466-1.11-1.466-.908-.62.069-.608.069-.608 1.003.07 1.531 1.032 1.531 1.032.892 1.53 2.341 1.088 2.91.832.092-.647.35-1.088.636-1.338-2.22-.253-4.555-1.113-4.555-4.951 0-1.093.39-1.988 1.029-2.688-.103-.253-.446-1.272.098-2.65 0 0 .84-.27 2.75 1.026A9.564 9.564 0 0112 6.844c.85.004 1.705.115 2.504.337 1.909-1.296 2.747-1.027 2.747-1.027.546 1.379.202 2.398.1 2.651.64.7 1.028 1.595 1.028 2.688 0 3.848-2.339 4.695-4.566 4.943.359.309.678.92.678 1.855 0 1.338-.012 2.419-.012 2.747 0 .268.18.58.688.482A10.019 10.019 0 0022 12.017C22 6.484 17.522 2 12 2z" clipRule="evenodd" />
    </svg>
  )
}
