#!/usr/bin/env bash
# Generate argparse parser from specification

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
TEMPLATES_DIR="$(dirname "$SCRIPT_DIR")/templates"

usage() {
    cat <<EOF
Generate argparse parser from specification

Usage: $(basename "$0") [OPTIONS]

Options:
    -n, --name NAME          Parser name (required)
    -d, --description DESC   Parser description
    -s, --subcommands        Include subcommands
    -c, --choices            Include choice validation
    -g, --groups             Include argument groups
    -o, --output FILE        Output file (default: stdout)
    -h, --help               Show this help

Examples:
    $(basename "$0") -n mycli -d "My CLI tool" -o mycli.py
    $(basename "$0") -n deploy -s -c -o deploy.py
EOF
    exit 1
}

# Parse arguments
NAME=""
DESCRIPTION=""
SUBCOMMANDS=false
CHOICES=false
GROUPS=false
OUTPUT=""

while [[ $# -gt 0 ]]; do
    case $1 in
        -n|--name)
            NAME="$2"
            shift 2
            ;;
        -d|--description)
            DESCRIPTION="$2"
            shift 2
            ;;
        -s|--subcommands)
            SUBCOMMANDS=true
            shift
            ;;
        -c|--choices)
            CHOICES=true
            shift
            ;;
        -g|--groups)
            GROUPS=true
            shift
            ;;
        -o|--output)
            OUTPUT="$2"
            shift 2
            ;;
        -h|--help)
            usage
            ;;
        *)
            echo "Error: Unknown option $1"
            usage
            ;;
    esac
done

if [ -z "$NAME" ]; then
    echo "Error: --name is required"
    usage
fi

# Set defaults
DESCRIPTION="${DESCRIPTION:-$NAME CLI tool}"

# Generate parser
generate_parser() {
    cat <<EOF
#!/usr/bin/env python3
"""
$DESCRIPTION

Generated by generate-parser.sh
"""

import argparse
import sys


def main():
    parser = argparse.ArgumentParser(
        description='$DESCRIPTION',
        formatter_class=argparse.RawDescriptionHelpFormatter
    )

    parser.add_argument(
        '--version',
        action='version',
        version='1.0.0'
    )

    parser.add_argument(
        '--verbose', '-v',
        action='store_true',
        help='Enable verbose output'
    )

EOF

    if [ "$GROUPS" = true ]; then
        cat <<EOF
    # Configuration group
    config_group = parser.add_argument_group(
        'configuration',
        'Configuration options'
    )

    config_group.add_argument(
        '--config',
        help='Configuration file'
    )

EOF
    fi

    if [ "$SUBCOMMANDS" = true ]; then
        cat <<EOF
    # Create subparsers
    subparsers = parser.add_subparsers(
        dest='command',
        help='Available commands',
        required=True
    )

    # Example subcommand
    cmd_parser = subparsers.add_parser(
        'run',
        help='Run the application'
    )

    cmd_parser.add_argument(
        'target',
        help='Target to run'
    )

EOF

        if [ "$CHOICES" = true ]; then
            cat <<EOF
    cmd_parser.add_argument(
        '--env',
        choices=['development', 'staging', 'production'],
        default='development',
        help='Environment (default: %(default)s)'
    )

EOF
        fi
    else
        cat <<EOF
    # Arguments
    parser.add_argument(
        'target',
        help='Target to process'
    )

EOF

        if [ "$CHOICES" = true ]; then
            cat <<EOF
    parser.add_argument(
        '--env',
        choices=['development', 'staging', 'production'],
        default='development',
        help='Environment (default: %(default)s)'
    )

EOF
        fi
    fi

    cat <<EOF
    # Parse arguments
    args = parser.parse_args()

    # Display configuration
    if args.verbose:
        print("Verbose mode enabled")
        print(f"Arguments: {args}")

    return 0


if __name__ == '__main__':
    sys.exit(main())
EOF
}

# Output
if [ -n "$OUTPUT" ]; then
    generate_parser > "$OUTPUT"
    chmod +x "$OUTPUT"
    echo "Generated parser: $OUTPUT"
else
    generate_parser
fi
