"""
Async SQLAlchemy session usage patterns and best practices
Examples of common database operations with proper async/await
"""

from typing import AsyncGenerator
from uuid import UUID
from sqlalchemy import select, update, delete, func, and_, or_
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy.orm import selectinload, joinedload, subqueryload

# Assume these are imported from your models
# from app.models.user import User, Post, Role
# from app.core.database import get_db, AsyncSessionLocal


# ============================================================================
# Pattern 1: FastAPI Dependency Injection (Recommended)
# ============================================================================


async def get_user_with_posts_dependency(
    user_id: UUID,
    db: AsyncSession,  # Injected by FastAPI via Depends(get_db)
) -> dict:
    """
    FastAPI endpoint pattern with dependency injection
    Session is automatically managed by FastAPI
    """
    stmt = select(User).where(User.id == user_id).options(selectinload(User.posts))

    result = await db.execute(stmt)
    user = result.scalar_one_or_none()

    if not user:
        return None

    return {
        "user": user,
        "posts_count": len(user.posts),
        "posts": user.posts,
    }


# ============================================================================
# Pattern 2: Manual Session Management with Context Manager
# ============================================================================


async def create_user_manual() -> User:
    """
    Manual session management using context manager
    Use when NOT in FastAPI endpoint context
    """
    async with AsyncSessionLocal() as session:
        try:
            user = User(
                email="user@example.com",
                username="johndoe",
                hashed_password="hashed_password_here",
                full_name="John Doe",
            )

            session.add(user)
            await session.commit()
            await session.refresh(user)

            return user

        except Exception as e:
            await session.rollback()
            raise


# ============================================================================
# Pattern 3: Transaction Management with Nested Transactions
# ============================================================================


async def transfer_post_ownership(
    session: AsyncSession,
    post_id: UUID,
    new_owner_id: UUID,
) -> bool:
    """
    Nested transaction pattern for complex operations
    """
    async with session.begin_nested():  # Creates a savepoint
        # Update post ownership
        stmt = (
            update(Post)
            .where(Post.id == post_id)
            .values(author_id=new_owner_id)
            .returning(Post)
        )

        result = await session.execute(stmt)
        post = result.scalar_one_or_none()

        if not post:
            raise ValueError("Post not found")

        # Update related statistics (example)
        # If this fails, rollback to savepoint
        old_user_stmt = (
            update(User)
            .where(User.id == post.author_id)
            .values(metadata=func.jsonb_set(User.metadata, "{posts_count}", "0"))
        )
        await session.execute(old_user_stmt)

    await session.commit()
    return True


# ============================================================================
# Pattern 4: Bulk Operations
# ============================================================================


async def bulk_create_users(session: AsyncSession, users_data: list[dict]) -> list[User]:
    """
    Efficient bulk insert pattern
    """
    users = [User(**data) for data in users_data]

    # Add all at once
    session.add_all(users)
    await session.commit()

    # Refresh to get database-generated values
    for user in users:
        await session.refresh(user)

    return users


async def bulk_update_status(session: AsyncSession, user_ids: list[UUID]) -> int:
    """
    Bulk update pattern
    Returns number of rows updated
    """
    stmt = update(User).where(User.id.in_(user_ids)).values(is_active=True)

    result = await session.execute(stmt)
    await session.commit()

    return result.rowcount


# ============================================================================
# Pattern 5: Eager Loading Strategies
# ============================================================================


async def get_users_with_relationships(session: AsyncSession) -> list[User]:
    """
    Different eager loading strategies for different relationships
    """
    stmt = (
        select(User)
        .options(
            # selectinload: Good for one-to-many (separate query)
            selectinload(User.posts),
            # joinedload: Good for many-to-one (SQL JOIN)
            joinedload(User.roles),
            # Nested loading
            selectinload(User.posts).selectinload(Post.comments),
        )
        .where(User.is_active == True)
        .limit(100)
    )

    result = await session.execute(stmt)
    # unique() required when using joinedload!
    return result.scalars().unique().all()


async def get_user_minimal_joins(session: AsyncSession, user_id: UUID) -> User:
    """
    Minimal eager loading for better performance
    Only load what you need
    """
    stmt = (
        select(User)
        .where(User.id == user_id)
        # Only load roles, skip posts
        .options(selectinload(User.roles))
    )

    result = await session.execute(stmt)
    return result.scalar_one()


# ============================================================================
# Pattern 6: Complex Queries with Aggregations
# ============================================================================


async def get_users_with_post_count(session: AsyncSession) -> list[dict]:
    """
    Query with aggregation and grouping
    """
    stmt = (
        select(User.id, User.username, func.count(Post.id).label("post_count"))
        .join(Post, User.id == Post.author_id, isouter=True)
        .group_by(User.id, User.username)
        .having(func.count(Post.id) > 0)
        .order_by(func.count(Post.id).desc())
    )

    result = await session.execute(stmt)

    return [
        {"user_id": row.id, "username": row.username, "post_count": row.post_count}
        for row in result
    ]


async def get_top_authors(session: AsyncSession, limit: int = 10) -> list[dict]:
    """
    Subquery pattern for complex filtering
    """
    # Subquery to get average views per user
    avg_views_subq = (
        select(Post.author_id, func.avg(Post.views).label("avg_views"))
        .group_by(Post.author_id)
        .subquery()
    )

    stmt = (
        select(
            User.id,
            User.username,
            func.count(Post.id).label("post_count"),
            avg_views_subq.c.avg_views,
        )
        .join(Post)
        .join(avg_views_subq, User.id == avg_views_subq.c.author_id)
        .group_by(User.id, User.username, avg_views_subq.c.avg_views)
        .order_by(avg_views_subq.c.avg_views.desc())
        .limit(limit)
    )

    result = await session.execute(stmt)
    return [
        {
            "user_id": row.id,
            "username": row.username,
            "post_count": row.post_count,
            "avg_views": float(row.avg_views),
        }
        for row in result
    ]


# ============================================================================
# Pattern 7: Streaming Large Results
# ============================================================================


async def stream_all_users(session: AsyncSession) -> AsyncGenerator[list[User], None]:
    """
    Stream large datasets in chunks to avoid memory issues
    """
    stmt = select(User).execution_options(yield_per=100)

    # Stream results
    stream = await session.stream(stmt)

    async for partition in stream.partitions(100):
        users = [row[0] for row in partition]
        yield users


# ============================================================================
# Pattern 8: Soft Delete Queries
# ============================================================================


async def get_active_users(
    session: AsyncSession,
    include_deleted: bool = False,
) -> list[User]:
    """
    Query with soft delete filtering
    """
    stmt = select(User)

    if not include_deleted:
        stmt = stmt.where(User.is_deleted == False)

    stmt = stmt.where(User.is_active == True).order_by(User.created_at.desc())

    result = await session.execute(stmt)
    return result.scalars().all()


async def soft_delete_user(session: AsyncSession, user_id: UUID) -> bool:
    """
    Soft delete pattern
    """
    from datetime import datetime

    stmt = (
        update(User)
        .where(User.id == user_id)
        .values(is_deleted=True, deleted_at=datetime.utcnow())
    )

    result = await session.execute(stmt)
    await session.commit()

    return result.rowcount > 0


# ============================================================================
# Pattern 9: Error Handling and Retries
# ============================================================================


from asyncio import sleep


async def create_user_with_retry(
    session: AsyncSession,
    user_data: dict,
    max_retries: int = 3,
) -> User | None:
    """
    Retry pattern for handling transient failures
    """
    for attempt in range(max_retries):
        try:
            user = User(**user_data)
            session.add(user)
            await session.commit()
            await session.refresh(user)
            return user

        except Exception as e:
            await session.rollback()

            if attempt < max_retries - 1:
                # Exponential backoff
                wait_time = 2**attempt
                await sleep(wait_time)
                continue
            else:
                # Final attempt failed
                raise


# ============================================================================
# Pattern 10: Query Result Transformation
# ============================================================================


async def get_users_as_dict(session: AsyncSession) -> list[dict]:
    """
    Transform query results to dictionaries
    """
    stmt = select(User).where(User.is_active == True).limit(100)

    result = await session.execute(stmt)
    users = result.scalars().all()

    return [
        {
            "id": str(user.id),
            "email": user.email,
            "username": user.username,
            "full_name": user.full_name,
            "is_verified": user.is_verified,
            "created_at": user.created_at.isoformat(),
        }
        for user in users
    ]


# ============================================================================
# Pattern 11: Pagination
# ============================================================================


async def get_paginated_users(
    session: AsyncSession,
    page: int = 1,
    page_size: int = 20,
) -> dict:
    """
    Pagination pattern with total count
    """
    # Get total count
    count_stmt = select(func.count(User.id)).where(User.is_deleted == False)
    total_result = await session.execute(count_stmt)
    total = total_result.scalar()

    # Get paginated results
    offset = (page - 1) * page_size

    stmt = (
        select(User)
        .where(User.is_deleted == False)
        .offset(offset)
        .limit(page_size)
        .order_by(User.created_at.desc())
    )

    result = await session.execute(stmt)
    users = result.scalars().all()

    return {
        "users": users,
        "total": total,
        "page": page,
        "page_size": page_size,
        "total_pages": (total + page_size - 1) // page_size,
    }


# ============================================================================
# Pattern 12: Dynamic Filtering
# ============================================================================


async def search_users(
    session: AsyncSession,
    email: str | None = None,
    username: str | None = None,
    is_active: bool | None = None,
) -> list[User]:
    """
    Dynamic query building based on provided filters
    """
    stmt = select(User).where(User.is_deleted == False)

    # Add filters dynamically
    if email:
        stmt = stmt.where(User.email.ilike(f"%{email}%"))

    if username:
        stmt = stmt.where(User.username.ilike(f"%{username}%"))

    if is_active is not None:
        stmt = stmt.where(User.is_active == is_active)

    result = await session.execute(stmt)
    return result.scalars().all()
