"""
Advanced query optimization patterns for async SQLAlchemy
Performance tips and anti-patterns to avoid
"""

from sqlalchemy import select, func, case, literal_column, cast, Integer
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy.orm import selectinload, joinedload, Load

# ============================================================================
# Query Optimization Patterns
# ============================================================================


async def avoid_n_plus_1_query(session: AsyncSession):
    """
    ❌ BAD: N+1 query problem
    ✅ GOOD: Eager loading with selectinload
    """

    # ❌ BAD: This will execute N+1 queries
    # stmt = select(User)
    # result = await session.execute(stmt)
    # users = result.scalars().all()
    # for user in users:
    #     print(user.posts)  # Separate query for each user!

    # ✅ GOOD: Single additional query for all posts
    stmt = select(User).options(selectinload(User.posts))
    result = await session.execute(stmt)
    users = result.scalars().all()

    for user in users:
        print(user.posts)  # No additional queries!

    return users


async def optimize_relationship_loading(session: AsyncSession):
    """
    Choose the right loading strategy for each relationship type
    """

    stmt = select(User).options(
        # selectinload: Best for one-to-many
        # Executes separate query with IN clause
        selectinload(User.posts),
        # joinedload: Best for many-to-one
        # Uses SQL JOIN - more efficient for single related object
        joinedload(User.posts).joinedload(Post.author),
        # Load only specific columns to reduce data transfer
        Load(User).load_only(User.id, User.username, User.email),
    )

    result = await session.execute(stmt)
    return result.scalars().unique().all()  # unique() required with joinedload


async def use_partial_column_loading(session: AsyncSession):
    """
    Load only needed columns to improve performance
    """

    # Load specific columns only
    stmt = select(User.id, User.email, User.username).where(User.is_active == True)

    result = await session.execute(stmt)
    # Returns tuples, not User objects
    return result.all()


async def optimize_exists_queries(session: AsyncSession, email: str):
    """
    Use efficient EXISTS queries instead of counting
    """

    # ❌ BAD: Counts all matching rows
    # stmt = select(func.count(User.id)).where(User.email == email)
    # count = (await session.execute(stmt)).scalar()
    # exists = count > 0

    # ✅ GOOD: Stops at first match
    stmt = select(User.id).where(User.email == email).limit(1)
    result = await session.execute(stmt)
    exists = result.first() is not None

    return exists


async def use_query_caching(session: AsyncSession):
    """
    Enable query result caching for frequently accessed data
    """

    # Cache query results
    stmt = (
        select(User)
        .where(User.is_active == True)
        .execution_options(compiled_cache={})  # Custom cache
    )

    result = await session.execute(stmt)
    return result.scalars().all()


# ============================================================================
# Batch Processing Patterns
# ============================================================================


async def batch_insert_efficient(session: AsyncSession, items: list[dict]):
    """
    Efficient batch insert using bulk operations
    """

    # ❌ BAD: Individual inserts in loop
    # for item in items:
    #     user = User(**item)
    #     session.add(user)
    #     await session.commit()  # Commits after each insert!

    # ✅ GOOD: Bulk insert with single commit
    users = [User(**item) for item in items]
    session.add_all(users)
    await session.commit()

    return len(users)


async def batch_update_with_returning(session: AsyncSession, user_ids: list[UUID]):
    """
    Batch update with RETURNING clause to get updated records
    """
    from sqlalchemy import update

    stmt = (
        update(User)
        .where(User.id.in_(user_ids))
        .values(is_active=True, updated_at=func.now())
        .returning(User)  # Returns updated records
    )

    result = await session.execute(stmt)
    await session.commit()

    return result.scalars().all()


# ============================================================================
# Index Usage Patterns
# ============================================================================


async def use_covering_index(session: AsyncSession):
    """
    Use composite indexes for better query performance
    """

    # This query benefits from index on (is_active, created_at)
    stmt = (
        select(User)
        .where(User.is_active == True)
        .order_by(User.created_at.desc())
        .limit(100)
    )

    result = await session.execute(stmt)
    return result.scalars().all()


async def use_partial_index_postgresql(session: AsyncSession):
    """
    Leverage PostgreSQL partial indexes for filtered queries
    """

    # This benefits from partial index:
    # CREATE INDEX ix_users_active ON users(email)
    # WHERE is_active = true AND is_deleted = false;

    stmt = select(User).where(
        User.email.like("%@example.com%"),
        User.is_active == True,
        User.is_deleted == False,
    )

    result = await session.execute(stmt)
    return result.scalars().all()


# ============================================================================
# Window Function Patterns
# ============================================================================


async def use_window_functions(session: AsyncSession):
    """
    Use window functions for advanced analytics
    """
    from sqlalchemy import over

    # Get row number within partition
    row_number = func.row_number().over(
        partition_by=Post.author_id, order_by=Post.created_at.desc()
    )

    stmt = (
        select(Post, row_number.label("row_num"))
        .where(User.id == Post.author_id)
        .order_by(Post.author_id, row_number)
    )

    result = await session.execute(stmt)

    # Filter in Python to get top N per author
    return [(row.Post, row.row_num) for row in result if row.row_num <= 5]


async def calculate_running_total(session: AsyncSession):
    """
    Calculate running totals with window functions
    """

    running_total = func.sum(Post.views).over(
        partition_by=Post.author_id, order_by=Post.created_at
    )

    stmt = select(
        Post.id, Post.title, Post.views, running_total.label("running_total")
    ).order_by(Post.author_id, Post.created_at)

    result = await session.execute(stmt)
    return result.all()


# ============================================================================
# JSON Query Patterns (PostgreSQL)
# ============================================================================


async def query_jsonb_field(session: AsyncSession):
    """
    Efficiently query JSONB fields in PostgreSQL
    """

    # Query JSON field
    stmt = select(User).where(User.metadata["preferences"]["theme"].astext == "dark")

    result = await session.execute(stmt)
    return result.scalars().all()


async def update_jsonb_field(session: AsyncSession, user_id: UUID):
    """
    Update specific keys in JSONB field
    """
    from sqlalchemy import update
    from sqlalchemy.dialects.postgresql import insert

    stmt = (
        update(User)
        .where(User.id == user_id)
        .values(
            metadata=func.jsonb_set(
                User.metadata,
                "{last_login}",
                cast(func.now().cast(String), JSONB),
            )
        )
    )

    await session.execute(stmt)
    await session.commit()


# ============================================================================
# Union and CTE Patterns
# ============================================================================


async def use_union_queries(session: AsyncSession):
    """
    Combine results from multiple queries with UNION
    """

    # Get active users
    active_stmt = select(User.id, User.email, literal_column("'active'").label("status")).where(
        User.is_active == True
    )

    # Get inactive users
    inactive_stmt = select(
        User.id, User.email, literal_column("'inactive'").label("status")
    ).where(User.is_active == False)

    # Combine with UNION
    union_stmt = active_stmt.union(inactive_stmt)

    result = await session.execute(union_stmt)
    return result.all()


async def use_cte_pattern(session: AsyncSession):
    """
    Use Common Table Expressions (CTE) for complex queries
    """

    # Create CTE
    active_users_cte = (
        select(User.id, User.username)
        .where(User.is_active == True)
        .cte("active_users")
    )

    # Use CTE in main query
    stmt = (
        select(Post, active_users_cte.c.username)
        .join(active_users_cte, Post.author_id == active_users_cte.c.id)
        .where(Post.is_published == True)
    )

    result = await session.execute(stmt)
    return result.all()


# ============================================================================
# Performance Monitoring
# ============================================================================


async def log_slow_queries(session: AsyncSession):
    """
    Enable query logging to identify slow queries
    """
    import logging

    # Enable echo to see all SQL queries
    # Set when creating engine: create_async_engine(url, echo=True)

    # Custom query timer
    import time

    start = time.time()

    stmt = select(User).options(selectinload(User.posts))
    result = await session.execute(stmt)
    users = result.scalars().all()

    duration = time.time() - start

    if duration > 1.0:  # Log queries taking > 1 second
        logging.warning(f"Slow query detected: {duration:.2f}s")

    return users
