"""
Complete User model example with relationships, indexes, and best practices
Demonstrates SQLAlchemy 2.0+ async patterns
"""

from datetime import datetime
from typing import TYPE_CHECKING
from uuid import UUID

from sqlalchemy import (
    String,
    Integer,
    Boolean,
    ForeignKey,
    Index,
    CheckConstraint,
    UniqueConstraint,
    text,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship
from sqlalchemy.dialects.postgresql import JSONB

# Import your base model
# from app.models.base import BaseModel

if TYPE_CHECKING:
    from .post_model import Post
    from .role_model import Role


class User(BaseModel):
    """
    User model with complete relationship examples
    Includes: one-to-many, many-to-many, indexes, constraints
    """

    __tablename__ = "users"

    # Basic fields with SQLAlchemy 2.0 Mapped syntax
    email: Mapped[str] = mapped_column(
        String(255),
        unique=True,
        nullable=False,
        index=True,
    )

    username: Mapped[str] = mapped_column(
        String(50),
        unique=True,
        nullable=False,
        index=True,
    )

    hashed_password: Mapped[str] = mapped_column(
        String(255),
        nullable=False,
    )

    full_name: Mapped[str | None] = mapped_column(
        String(255),
        nullable=True,
    )

    # Boolean fields
    is_active: Mapped[bool] = mapped_column(
        Boolean,
        default=True,
        nullable=False,
        index=True,
    )

    is_verified: Mapped[bool] = mapped_column(
        Boolean,
        default=False,
        nullable=False,
    )

    is_superuser: Mapped[bool] = mapped_column(
        Boolean,
        default=False,
        nullable=False,
    )

    # Integer field with default
    login_count: Mapped[int] = mapped_column(
        Integer,
        default=0,
        nullable=False,
    )

    # JSON field (PostgreSQL specific)
    metadata: Mapped[dict | None] = mapped_column(
        JSONB,
        nullable=True,
        server_default=text("'{}'::jsonb"),
    )

    # Timestamp fields (inherited from BaseModel, shown for reference)
    # created_at: Mapped[datetime]
    # updated_at: Mapped[datetime]

    # One-to-many relationship: User has many Posts
    posts: Mapped[list["Post"]] = relationship(
        back_populates="author",
        cascade="all, delete-orphan",
        lazy="selectin",  # Eager load by default
        order_by="Post.created_at.desc()",
    )

    # One-to-many: User has many comments
    comments: Mapped[list["Comment"]] = relationship(
        back_populates="author",
        cascade="all, delete-orphan",
        lazy="selectin",
    )

    # Many-to-many: User has many Roles
    roles: Mapped[list["Role"]] = relationship(
        secondary="user_roles",
        back_populates="users",
        lazy="selectin",
    )

    # Self-referential many-to-many: User follows Users
    following: Mapped[list["User"]] = relationship(
        secondary="user_follows",
        primaryjoin="User.id == user_follows.c.follower_id",
        secondaryjoin="User.id == user_follows.c.following_id",
        back_populates="followers",
        lazy="selectin",
    )

    followers: Mapped[list["User"]] = relationship(
        secondary="user_follows",
        primaryjoin="User.id == user_follows.c.following_id",
        secondaryjoin="User.id == user_follows.c.follower_id",
        back_populates="following",
        lazy="selectin",
    )

    # Table-level constraints and indexes
    __table_args__ = (
        # Composite index for common query patterns
        Index("ix_user_email_active", "email", "is_active"),
        Index("ix_user_username_active", "username", "is_active"),
        # Partial index (PostgreSQL) - only index active users
        Index(
            "ix_user_active_verified",
            "email",
            postgresql_where=text("is_active = true AND is_deleted = false"),
        ),
        # Check constraint
        CheckConstraint("login_count >= 0", name="check_login_count_positive"),
        # Unique constraint
        UniqueConstraint("email", "is_deleted", name="uq_user_email_active"),
    )

    def __repr__(self) -> str:
        return f"<User(id={self.id}, email={self.email}, username={self.username})>"

    # Custom methods
    async def increment_login(self, session) -> None:
        """Increment login counter"""
        self.login_count += 1
        await session.commit()

    async def verify_email(self, session) -> None:
        """Mark email as verified"""
        self.is_verified = True
        await session.commit()

    async def deactivate(self, session) -> None:
        """Deactivate user account"""
        self.is_active = False
        await session.commit()

    def has_role(self, role_name: str) -> bool:
        """Check if user has a specific role"""
        return any(role.name == role_name for role in self.roles)

    @property
    def is_admin(self) -> bool:
        """Check if user is admin"""
        return self.is_superuser or self.has_role("admin")


class Post(BaseModel):
    """Example Post model with many-to-one relationship"""

    __tablename__ = "posts"

    title: Mapped[str] = mapped_column(String(255), nullable=False)
    content: Mapped[str] = mapped_column(nullable=False)
    views: Mapped[int] = mapped_column(Integer, default=0, nullable=False)
    is_published: Mapped[bool] = mapped_column(Boolean, default=False, nullable=False)

    # Foreign key to User
    author_id: Mapped[UUID] = mapped_column(
        ForeignKey("users.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    # Many-to-one relationship
    author: Mapped["User"] = relationship(
        back_populates="posts",
        lazy="joined",  # Use JOIN for loading
    )

    # One-to-many
    comments: Mapped[list["Comment"]] = relationship(
        back_populates="post",
        cascade="all, delete-orphan",
        lazy="selectin",
    )

    __table_args__ = (
        Index("ix_post_author_published", "author_id", "is_published"),
        Index("ix_post_published_created", "is_published", "created_at"),
    )


class Comment(BaseModel):
    """Example Comment model"""

    __tablename__ = "comments"

    content: Mapped[str] = mapped_column(nullable=False)

    # Foreign keys
    author_id: Mapped[UUID] = mapped_column(
        ForeignKey("users.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    post_id: Mapped[UUID] = mapped_column(
        ForeignKey("posts.id", ondelete="CASCADE"),
        nullable=False,
        index=True,
    )

    # Relationships
    author: Mapped["User"] = relationship(back_populates="comments", lazy="joined")
    post: Mapped["Post"] = relationship(back_populates="comments", lazy="joined")


class Role(BaseModel):
    """Example Role model for many-to-many"""

    __tablename__ = "roles"

    name: Mapped[str] = mapped_column(String(50), unique=True, nullable=False)
    description: Mapped[str | None] = mapped_column(String(255), nullable=True)

    # Many-to-many back reference
    users: Mapped[list["User"]] = relationship(
        secondary="user_roles",
        back_populates="roles",
        lazy="selectin",
    )


# Association tables for many-to-many relationships
from sqlalchemy import Table, Column, ForeignKey

user_roles = Table(
    "user_roles",
    BaseModel.metadata,
    Column("user_id", ForeignKey("users.id", ondelete="CASCADE"), primary_key=True),
    Column("role_id", ForeignKey("roles.id", ondelete="CASCADE"), primary_key=True),
    Column("assigned_at", DateTime(timezone=True), server_default=func.now()),
)

user_follows = Table(
    "user_follows",
    BaseModel.metadata,
    Column(
        "follower_id", ForeignKey("users.id", ondelete="CASCADE"), primary_key=True
    ),
    Column(
        "following_id", ForeignKey("users.id", ondelete="CASCADE"), primary_key=True
    ),
    Column("created_at", DateTime(timezone=True), server_default=func.now()),
    # Prevent self-follows
    CheckConstraint("follower_id != following_id", name="check_no_self_follow"),
)
