#!/bin/bash
# Generate Alembic migration from model changes

set -e

PROJECT_ROOT="${2:-.}"
MESSAGE="${1:-Update database schema}"

if [ -z "$1" ]; then
    echo "Usage: $0 <migration_message> [project_root]"
    echo "Example: $0 'add user table' ."
    exit 1
fi

echo "Generating migration: $MESSAGE"

# Check if Alembic is initialized
if [ ! -d "$PROJECT_ROOT/alembic" ]; then
    echo "Error: Alembic not initialized. Run setup-alembic.sh first"
    exit 1
fi

# Check if database URL is set
if [ -z "$DATABASE_URL" ]; then
    if [ -f "$PROJECT_ROOT/.env" ]; then
        echo "Loading .env file..."
        export $(grep -v '^#' "$PROJECT_ROOT/.env" | xargs)
    fi
fi

# Generate migration
cd "$PROJECT_ROOT"
alembic revision --autogenerate -m "$MESSAGE"

# Get the latest migration file
LATEST_MIGRATION=$(ls -t alembic/versions/*.py | head -1)

echo ""
echo "Migration generated: $LATEST_MIGRATION"
echo ""
echo "IMPORTANT: Review the migration file before applying!"
echo ""
echo "Common checks:"
echo "  - Verify column types are correct"
echo "  - Check foreign key constraints"
echo "  - Ensure indexes are appropriate"
echo "  - Review default values"
echo "  - Test both upgrade() and downgrade()"
echo ""
echo "To apply migration:"
echo "  alembic upgrade head"
echo ""
echo "To rollback:"
echo "  alembic downgrade -1"
echo ""

# Display migration content
echo "Migration preview:"
echo "===================="
cat "$LATEST_MIGRATION"
echo "===================="
