#!/bin/bash
# Analyze SQLAlchemy queries for optimization opportunities

set -e

PROJECT_ROOT="${1:-.}"

echo "Analyzing queries for optimization opportunities..."
echo ""

# Search for common anti-patterns
echo "Checking for potential N+1 query issues..."
echo "=========================================="

# Look for relationship access without eager loading
grep -rn "\.scalars()" "$PROJECT_ROOT" --include="*.py" | \
    grep -v "selectinload\|joinedload\|subqueryload" | \
    head -10 || echo "No obvious N+1 patterns found"

echo ""
echo "Checking for missing indexes..."
echo "==============================="

# Look for filter operations without indexes
grep -rn "\.where\|\.filter" "$PROJECT_ROOT" --include="*.py" | \
    grep -v "id ==" | \
    head -10 || echo "Review WHERE clauses for index opportunities"

echo ""
echo "Checking for missing relationship lazy loading config..."
echo "========================================================"

# Look for relationships without explicit lazy config
grep -rn "relationship(" "$PROJECT_ROOT" --include="*.py" | \
    grep -v "lazy=" | \
    head -10 || echo "All relationships have explicit lazy configuration"

echo ""
echo "Recommendations:"
echo "================"
echo "1. Enable SQL echo during development to see generated queries:"
echo "   engine = create_async_engine(url, echo=True)"
echo ""
echo "2. Use eager loading for relationships:"
echo "   stmt = select(User).options(selectinload(User.posts))"
echo ""
echo "3. Add indexes for frequently filtered columns:"
echo "   column: Mapped[str] = mapped_column(index=True)"
echo ""
echo "4. Use bulk operations for large datasets:"
echo "   await session.execute(insert(User).values(users))"
echo ""
echo "5. Configure connection pooling appropriately:"
echo "   pool_size=20, max_overflow=40"
echo ""
