#!/bin/bash
# Setup Alembic with async SQLAlchemy support

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"
PROJECT_ROOT="${1:-.}"

echo "Setting up Alembic with async support..."

# Check if alembic is installed
if ! python -c "import alembic" 2>/dev/null; then
    echo "Installing alembic..."
    pip install alembic
fi

# Initialize Alembic if not already initialized
if [ ! -d "$PROJECT_ROOT/alembic" ]; then
    echo "Initializing Alembic..."
    cd "$PROJECT_ROOT"
    alembic init alembic
else
    echo "Alembic already initialized"
fi

# Copy async-compatible alembic.ini
if [ -f "$SKILL_DIR/templates/alembic.ini" ]; then
    echo "Copying async-compatible alembic.ini..."
    cp "$SKILL_DIR/templates/alembic.ini" "$PROJECT_ROOT/alembic.ini.template"
    echo "Template saved to alembic.ini.template - review and replace alembic.ini if needed"
fi

# Update env.py for async support
echo "Updating env.py for async support..."
cat > "$PROJECT_ROOT/alembic/env.py" << 'EOF'
from logging.config import fileConfig
from sqlalchemy import pool
from sqlalchemy.engine import Connection
from sqlalchemy.ext.asyncio import async_engine_from_config
from alembic import context
import asyncio

# Import your models here
from app.core.database import Base
from app.core.config import settings

# Alembic Config object
config = context.config

# Override sqlalchemy.url with environment variable
config.set_main_option("sqlalchemy.url", settings.DATABASE_URL)

# Interpret the config file for Python logging
if config.config_file_name is not None:
    fileConfig(config.config_file_name)

# Target metadata for autogenerate
target_metadata = Base.metadata


def run_migrations_offline() -> None:
    """Run migrations in 'offline' mode."""
    url = config.get_main_option("sqlalchemy.url")
    context.configure(
        url=url,
        target_metadata=target_metadata,
        literal_binds=True,
        dialect_opts={"paramstyle": "named"},
        compare_type=True,
        compare_server_default=True,
    )

    with context.begin_transaction():
        context.run_migrations()


def do_run_migrations(connection: Connection) -> None:
    context.configure(
        connection=connection,
        target_metadata=target_metadata,
        compare_type=True,
        compare_server_default=True,
    )

    with context.begin_transaction():
        context.run_migrations()


async def run_async_migrations() -> None:
    """Run migrations in 'online' mode with async support."""
    connectable = async_engine_from_config(
        config.get_section(config.config_ini_section, {}),
        prefix="sqlalchemy.",
        poolclass=pool.NullPool,
    )

    async with connectable.connect() as connection:
        await connection.run_sync(do_run_migrations)

    await connectable.dispose()


def run_migrations_online() -> None:
    """Run migrations in 'online' mode."""
    asyncio.run(run_async_migrations())


if context.is_offline_mode():
    run_migrations_offline()
else:
    run_migrations_online()
EOF

echo ""
echo "Alembic setup complete!"
echo ""
echo "Next steps:"
echo "1. Update DATABASE_URL in your .env or config"
echo "2. Import all models in alembic/env.py (line 11-12)"
echo "3. Generate first migration: alembic revision --autogenerate -m 'initial'"
echo "4. Review migration in alembic/versions/"
echo "5. Apply migration: alembic upgrade head"
echo ""
