#!/bin/bash
# Test async database connection

set -e

PROJECT_ROOT="${1:-.}"

echo "Testing database connection..."

# Load environment variables
if [ -f "$PROJECT_ROOT/.env" ]; then
    export $(grep -v '^#' "$PROJECT_ROOT/.env" | xargs)
fi

# Create test script
cat > /tmp/test_db_connection.py << 'EOF'
import asyncio
import sys
from sqlalchemy.ext.asyncio import create_async_engine
from sqlalchemy import text
import os

async def test_connection():
    database_url = os.getenv("DATABASE_URL")

    if not database_url:
        print("ERROR: DATABASE_URL not set")
        sys.exit(1)

    print(f"Testing connection to: {database_url.split('@')[1] if '@' in database_url else 'database'}")

    try:
        engine = create_async_engine(database_url, echo=False)

        async with engine.connect() as conn:
            result = await conn.execute(text("SELECT 1"))
            value = result.scalar()

            if value == 1:
                print("✓ Connection successful!")

                # Get database version
                if "postgresql" in database_url:
                    result = await conn.execute(text("SELECT version()"))
                    version = result.scalar()
                    print(f"✓ PostgreSQL version: {version.split(',')[0]}")
                elif "mysql" in database_url:
                    result = await conn.execute(text("SELECT VERSION()"))
                    version = result.scalar()
                    print(f"✓ MySQL version: {version}")
                elif "sqlite" in database_url:
                    result = await conn.execute(text("SELECT sqlite_version()"))
                    version = result.scalar()
                    print(f"✓ SQLite version: {version}")

        await engine.dispose()
        print("✓ All checks passed!")
        return 0

    except Exception as e:
        print(f"✗ Connection failed: {e}")
        return 1

if __name__ == "__main__":
    exit_code = asyncio.run(test_connection())
    sys.exit(exit_code)
EOF

# Run test
python /tmp/test_db_connection.py
EXIT_CODE=$?

# Cleanup
rm /tmp/test_db_connection.py

exit $EXIT_CODE
