"""
Base model template with common patterns for SQLAlchemy 2.0+
Includes UUID primary keys, timestamps, and soft delete support
"""

from datetime import datetime
from typing import Any
from uuid import UUID, uuid4

from sqlalchemy import DateTime, Boolean, func, select
from sqlalchemy.orm import Mapped, mapped_column, DeclarativeBase
from sqlalchemy.ext.asyncio import AsyncSession


class Base(DeclarativeBase):
    """Base class for all models"""

    pass


class UUIDMixin:
    """UUID primary key mixin"""

    id: Mapped[UUID] = mapped_column(
        primary_key=True,
        default=uuid4,
        nullable=False,
    )


class TimestampMixin:
    """Automatic timestamp management"""

    created_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True),
        server_default=func.now(),
        nullable=False,
    )

    updated_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True),
        server_default=func.now(),
        onupdate=func.now(),
        nullable=False,
    )


class SoftDeleteMixin:
    """Soft delete support"""

    is_deleted: Mapped[bool] = mapped_column(
        Boolean,
        default=False,
        nullable=False,
        index=True,
    )

    deleted_at: Mapped[datetime | None] = mapped_column(
        DateTime(timezone=True),
        nullable=True,
    )

    async def soft_delete(self, session: AsyncSession) -> None:
        """Mark record as deleted"""
        self.is_deleted = True
        self.deleted_at = datetime.utcnow()
        await session.commit()

    async def restore(self, session: AsyncSession) -> None:
        """Restore soft-deleted record"""
        self.is_deleted = False
        self.deleted_at = None
        await session.commit()


class BaseModel(Base, UUIDMixin, TimestampMixin, SoftDeleteMixin):
    """
    Complete base model with all mixins
    Use this as parent for most models
    """

    __abstract__ = True

    def dict(self) -> dict[str, Any]:
        """Convert model to dictionary"""
        return {
            column.name: getattr(self, column.name)
            for column in self.__table__.columns
        }

    def update(self, **kwargs: Any) -> None:
        """Update model attributes"""
        for key, value in kwargs.items():
            if hasattr(self, key):
                setattr(self, key, value)

    @classmethod
    async def get_by_id(
        cls, session: AsyncSession, id: UUID
    ) -> "BaseModel | None":
        """Get record by ID"""
        return await session.get(cls, id)

    @classmethod
    async def get_all(
        cls,
        session: AsyncSession,
        skip: int = 0,
        limit: int = 100,
        include_deleted: bool = False,
    ) -> list["BaseModel"]:
        """Get all records with pagination"""
        stmt = select(cls)

        if not include_deleted:
            stmt = stmt.where(cls.is_deleted == False)

        stmt = stmt.offset(skip).limit(limit).order_by(cls.created_at.desc())

        result = await session.execute(stmt)
        return list(result.scalars().all())

    @classmethod
    async def count(
        cls, session: AsyncSession, include_deleted: bool = False
    ) -> int:
        """Count records"""
        stmt = select(func.count(cls.id))

        if not include_deleted:
            stmt = stmt.where(cls.is_deleted == False)

        result = await session.execute(stmt)
        return result.scalar_one()

    async def save(self, session: AsyncSession) -> None:
        """Save the model"""
        session.add(self)
        await session.commit()
        await session.refresh(self)

    async def delete(self, session: AsyncSession, hard: bool = False) -> None:
        """
        Delete the model
        Args:
            session: Database session
            hard: If True, perform hard delete. Otherwise soft delete.
        """
        if hard:
            await session.delete(self)
            await session.commit()
        else:
            await self.soft_delete(session)

    def __repr__(self) -> str:
        """String representation"""
        return f"<{self.__class__.__name__}(id={self.id})>"


# Alternative: Simple base without soft delete
class SimpleBaseModel(Base, UUIDMixin, TimestampMixin):
    """Base model without soft delete functionality"""

    __abstract__ = True

    def dict(self) -> dict[str, Any]:
        """Convert model to dictionary"""
        return {
            column.name: getattr(self, column.name)
            for column in self.__table__.columns
        }
