"""
Async session management for FastAPI with SQLAlchemy 2.0+
Includes connection pooling, dependencies, and context managers
"""

from typing import AsyncGenerator
from contextlib import asynccontextmanager

from sqlalchemy.ext.asyncio import (
    AsyncSession,
    AsyncEngine,
    create_async_engine,
    async_sessionmaker,
)
from sqlalchemy.orm import declarative_base
from sqlalchemy.pool import NullPool, QueuePool

# Import your config (adjust path as needed)
# from app.core.config import settings


class DatabaseConfig:
    """Database configuration settings"""

    # Replace with your actual config
    DATABASE_URL: str = "postgresql+asyncpg://user:pass@localhost/dbname"
    DEBUG: bool = False
    POOL_SIZE: int = 5
    MAX_OVERFLOW: int = 10
    POOL_RECYCLE: int = 3600
    POOL_PRE_PING: bool = True
    POOL_TIMEOUT: int = 30


settings = DatabaseConfig()


class DatabaseSessionManager:
    """
    Centralized database session manager
    Handles engine creation and session lifecycle
    """

    def __init__(self, database_url: str, echo: bool = False):
        """
        Initialize session manager

        Args:
            database_url: Database connection URL
            echo: Enable SQL query logging
        """
        self._engine: AsyncEngine | None = None
        self._sessionmaker: async_sessionmaker[AsyncSession] | None = None
        self._database_url = database_url
        self._echo = echo

    def init(self, **engine_kwargs) -> None:
        """
        Initialize the database engine and session maker

        Common engine_kwargs:
            pool_size: Number of connections to maintain
            max_overflow: Additional connections allowed
            pool_recycle: Recycle connections after N seconds
            pool_pre_ping: Verify connection health before use
            pool_timeout: Seconds to wait for connection
        """
        if self._engine is not None:
            return

        # Determine poolclass based on database type
        poolclass = QueuePool
        if "sqlite" in self._database_url:
            # SQLite doesn't support connection pooling
            poolclass = NullPool

        self._engine = create_async_engine(
            self._database_url,
            echo=self._echo,
            poolclass=poolclass,
            pool_size=engine_kwargs.get("pool_size", settings.POOL_SIZE),
            max_overflow=engine_kwargs.get("max_overflow", settings.MAX_OVERFLOW),
            pool_recycle=engine_kwargs.get("pool_recycle", settings.POOL_RECYCLE),
            pool_pre_ping=engine_kwargs.get("pool_pre_ping", settings.POOL_PRE_PING),
            pool_timeout=engine_kwargs.get("pool_timeout", settings.POOL_TIMEOUT),
        )

        self._sessionmaker = async_sessionmaker(
            self._engine,
            class_=AsyncSession,
            expire_on_commit=False,  # Keep objects usable after commit
            autoflush=False,  # Manual control over flushing
            autocommit=False,  # Explicit transaction control
        )

    async def close(self) -> None:
        """Close database engine and all connections"""
        if self._engine is None:
            return

        await self._engine.dispose()
        self._engine = None
        self._sessionmaker = None

    @asynccontextmanager
    async def session(self) -> AsyncGenerator[AsyncSession, None]:
        """
        Context manager for database sessions
        Automatically commits on success, rolls back on error

        Usage:
            async with sessionmanager.session() as session:
                result = await session.execute(stmt)
        """
        if self._sessionmaker is None:
            raise RuntimeError("DatabaseSessionManager is not initialized")

        async with self._sessionmaker() as session:
            try:
                yield session
                await session.commit()
            except Exception:
                await session.rollback()
                raise
            finally:
                await session.close()

    async def get_session(self) -> AsyncGenerator[AsyncSession, None]:
        """
        FastAPI dependency for getting database sessions

        Usage in FastAPI:
            @app.get("/users")
            async def get_users(db: AsyncSession = Depends(get_db)):
                ...
        """
        async with self.session() as session:
            yield session

    @property
    def engine(self) -> AsyncEngine:
        """Get the database engine"""
        if self._engine is None:
            raise RuntimeError("DatabaseSessionManager is not initialized")
        return self._engine


# Global session manager instance
sessionmanager = DatabaseSessionManager(
    database_url=settings.DATABASE_URL,
    echo=settings.DEBUG,
)


# Initialize on import (or move to app startup)
sessionmanager.init()


# FastAPI dependency
async def get_db() -> AsyncGenerator[AsyncSession, None]:
    """
    FastAPI dependency for database sessions

    Usage:
        from app.core.database import get_db

        @app.get("/users")
        async def get_users(db: AsyncSession = Depends(get_db)):
            result = await db.execute(select(User))
            return result.scalars().all()
    """
    async with sessionmanager.session() as session:
        yield session


# Alternative: Simple session factory without manager
def create_session_factory(database_url: str, **kwargs) -> async_sessionmaker[AsyncSession]:
    """
    Create a session factory directly

    Args:
        database_url: Database connection URL
        **kwargs: Additional engine arguments

    Returns:
        Session factory
    """
    engine = create_async_engine(database_url, **kwargs)

    return async_sessionmaker(
        engine,
        class_=AsyncSession,
        expire_on_commit=False,
        autoflush=False,
        autocommit=False,
    )


# Base declarative class
Base = declarative_base()


# Lifespan management for FastAPI
async def init_db() -> None:
    """Initialize database connection (call on app startup)"""
    sessionmanager.init()


async def close_db() -> None:
    """Close database connection (call on app shutdown)"""
    await sessionmanager.close()


# Example FastAPI lifespan context manager
from contextlib import asynccontextmanager
from typing import AsyncIterator


@asynccontextmanager
async def lifespan(app) -> AsyncIterator[None]:
    """
    FastAPI lifespan context manager

    Usage in FastAPI app:
        from app.core.database import lifespan

        app = FastAPI(lifespan=lifespan)
    """
    # Startup
    await init_db()
    yield
    # Shutdown
    await close_db()
