#!/bin/bash
# Consume remote A2A agent
# Usage: bash consume-agent.sh --url <agent-url>

set -e

# Parse arguments
AGENT_URL=""
OUTPUT_FILE="a2a_client.py"

while [[ $# -gt 0 ]]; do
  case $1 in
    --url)
      AGENT_URL="$2"
      shift 2
      ;;
    --output)
      OUTPUT_FILE="$2"
      shift 2
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

# Validate required arguments
if [ -z "$AGENT_URL" ]; then
  echo "Error: --url is required"
  echo "Usage: bash consume-agent.sh --url <agent-url> [--output <file>]"
  exit 1
fi

echo "Resolving Agent Card from: $AGENT_URL"

# Fetch Agent Card
AGENT_CARD_URL="${AGENT_URL%/}/.well-known/agent.json"
AGENT_CARD=$(curl -s "$AGENT_CARD_URL")

if [ -z "$AGENT_CARD" ]; then
  echo "Error: Could not fetch Agent Card from $AGENT_CARD_URL"
  exit 1
fi

echo "Agent Card retrieved successfully"

# Extract agent details
AGENT_NAME=$(echo "$AGENT_CARD" | jq -r '.name')
AGENT_DESC=$(echo "$AGENT_CARD" | jq -r '.description')
AGENT_ENDPOINT=$(echo "$AGENT_CARD" | jq -r '.url')

echo "Agent Name: $AGENT_NAME"
echo "Description: $AGENT_DESC"
echo "Endpoint: $AGENT_ENDPOINT"

# Generate client code
cat > "$OUTPUT_FILE" << EOF
"""
A2A Client for: $AGENT_NAME
Description: $AGENT_DESC
Endpoint: $AGENT_ENDPOINT

Auto-generated by consume-agent.sh
"""

from adk import Agent
from a2a import A2ACardResolver, send_task
import asyncio


async def use_${AGENT_NAME//-/_}_agent():
    """Use the remote $AGENT_NAME agent via A2A protocol"""

    # Resolve Agent Card
    resolver = A2ACardResolver()
    agent_card = await resolver.resolve("$AGENT_ENDPOINT")

    print(f"Connected to agent: {agent_card.name}")
    print(f"Capabilities: {agent_card.capabilities}")

    # Create send_task tool
    send_task_tool = send_task(
        agent_url=agent_card.endpoint,
        session_id="session-$(uuidgen)"
    )

    # Create your agent that uses the remote agent
    my_agent = Agent(
        name="my-agent",
        tools=[send_task_tool],
        instructions="""
        You can communicate with the remote $AGENT_NAME agent.
        Use the send_task tool to delegate tasks to it.
        """
    )

    # Example usage
    result = await my_agent.run("Delegate a task to the remote agent")
    return result


if __name__ == "__main__":
    result = asyncio.run(use_${AGENT_NAME//-/_}_agent())
    print(result)
EOF

echo "Client code generated: $OUTPUT_FILE"
echo ""
echo "Next steps:"
echo "1. Review the generated code in $OUTPUT_FILE"
echo "2. Customize the agent instructions and tools"
echo "3. Run: python $OUTPUT_FILE"
