# Google Cloud Platform ML Configuration Skill

Complete configuration templates, scripts, and examples for BigQuery ML and Vertex AI training on Google Cloud Platform.

## Overview

This skill provides everything needed to set up and run machine learning training on GCP:

- **BigQuery ML**: SQL-based machine learning for structured data
- **Vertex AI**: Custom training with GPUs/TPUs for deep learning
- **Authentication**: Secure GCP credential management
- **Cost Estimation**: Calculate training costs before running jobs

## Quick Start

### 1. Setup BigQuery ML

```bash
bash scripts/setup-bigquery-ml.sh
```

This will:
- Authenticate with GCP
- Enable required APIs
- Create BigQuery dataset
- Generate example SQL queries

### 2. Setup Vertex AI Training

```bash
bash scripts/setup-vertex-ai.sh
```

This will:
- Configure Vertex AI project
- Create GCS bucket
- Set up GPU/TPU configuration
- Generate training templates

### 3. Configure Authentication

```bash
bash scripts/configure-auth.sh
```

Choose from:
- User account (local development)
- Service account (automation)
- Workload Identity (GKE)

### 4. Estimate Costs

```bash
bash scripts/estimate-gcp-cost.sh
```

Get cost estimates for:
- BigQuery ML training
- Vertex AI GPU training
- Vertex AI TPU training
- Storage and data transfer

## Directory Structure

```
google-cloud-configs/
├── SKILL.md                 # Skill documentation
├── README.md                # This file
├── scripts/                 # Setup and utility scripts
│   ├── setup-bigquery-ml.sh
│   ├── setup-vertex-ai.sh
│   ├── configure-auth.sh
│   └── estimate-gcp-cost.sh
├── templates/              # Configuration templates
│   ├── bigquery_ml_training.sql      # SQL training templates
│   ├── vertex_training_job.py        # PyTorch training template
│   ├── vertex_gpu_config.yaml        # GPU configurations
│   ├── vertex_tpu_config.yaml        # TPU configurations
│   └── gcp_auth.json                 # Auth template
└── examples/               # Working examples
    ├── bigquery-regression-example.sql      # NYC taxi regression
    ├── vertex-pytorch-training.py           # IMDB sentiment analysis
    └── vertex-huggingface-finetuning.py     # HF Trainer example
```

## Usage Examples

### BigQuery ML: Taxi Trip Duration Prediction

```sql
-- Load example
cat examples/bigquery-regression-example.sql

-- Run in BigQuery console or CLI
bq query --use_legacy_sql=false < examples/bigquery-regression-example.sql
```

Features:
- Real public dataset (NYC taxi trips)
- Complete feature engineering
- XGBoost model training
- Evaluation and predictions
- Cost: ~$0.05

### Vertex AI: Sentiment Analysis

```bash
# Local testing
python examples/vertex-pytorch-training.py --local

# Submit to Vertex AI
gcloud ai custom-jobs create \
  --region=us-central1 \
  --display-name=sentiment-training \
  --config=templates/vertex_gpu_config.yaml
```

Features:
- DistilBERT fine-tuning
- Mixed precision training
- GCS checkpointing
- Vertex AI metrics logging

### Hugging Face Fine-tuning

```bash
python examples/vertex-huggingface-finetuning.py \
  --output_dir=./output \
  --num_train_epochs=3 \
  --per_device_train_batch_size=16 \
  --fp16
```

Features:
- Trainer API integration
- Automatic evaluation
- Early stopping
- Model deployment

## GPU/TPU Selection Guide

### When to use GPUs

- PyTorch/custom frameworks
- Small to medium models
- Flexible batch sizes
- Frequent code changes

**Recommended GPUs:**
- **T4**: Prototyping ($0.35/hr)
- **A100**: Standard training ($3.67/hr)
- **L4**: Cost-effective modern GPU ($0.66/hr)

### When to use TPUs

- TensorFlow/JAX workloads
- Large batch sizes (>1024)
- Matrix-heavy operations
- Production scale training

**Recommended TPUs:**
- **v5e-8**: Development ($2.50/hr)
- **v3-8**: Standard training ($8/hr)
- **v4-8**: Latest generation ($11/hr)

## Cost Optimization Tips

### BigQuery ML

1. **Use partitioned tables** - Reduce data scanned
2. **Filter before training** - WHERE clause optimization
3. **Start simple** - LINEAR_REG before AutoML
4. **Sample for testing** - Use LIMIT for experiments

### Vertex AI

1. **Use preemptible instances** - 60% cost reduction
2. **Enable checkpointing** - Resume interrupted jobs
3. **Mixed precision** - Faster = cheaper
4. **Right-size GPUs** - Don't over-provision
5. **Clean up storage** - Delete old artifacts

## Security Best Practices

### Credentials

- ✅ Use service accounts with minimal permissions
- ✅ Store keys in Secret Manager
- ✅ Add `*.json` to `.gitignore`
- ✅ Rotate keys every 90 days
- ❌ Never commit credentials to git
- ❌ Never hardcode API keys

### IAM Roles

**BigQuery ML:**
- `roles/bigquery.dataEditor`
- `roles/bigquery.jobUser`

**Vertex AI:**
- `roles/aiplatform.user`
- `roles/storage.objectAdmin`

## Common Workflows

### Workflow 1: Quick BigQuery ML Prototype

```bash
# 1. Setup
bash scripts/setup-bigquery-ml.sh

# 2. Copy example
cp templates/bigquery_ml_training.sql my_model.sql

# 3. Edit for your data
# ... modify SQL ...

# 4. Train
bq query --use_legacy_sql=false < my_model.sql

# 5. Evaluate
bq query "SELECT * FROM ML.EVALUATE(MODEL project.dataset.my_model)"
```

**Time:** 30 minutes
**Cost:** $0.05-5.00 depending on data size

### Workflow 2: Custom PyTorch on Vertex AI

```bash
# 1. Setup
bash scripts/setup-vertex-ai.sh
bash scripts/configure-auth.sh

# 2. Prepare training script
cp templates/vertex_training_job.py train.py
# ... customize model and dataset ...

# 3. Test locally
python train.py --local --num_epochs=1

# 4. Submit to Vertex AI
gcloud ai custom-jobs create \
  --region=us-central1 \
  --config=templates/vertex_gpu_config.yaml

# 5. Monitor
gcloud ai custom-jobs stream-logs JOB_ID
```

**Time:** 1-2 hours
**Cost:** $1-20 depending on GPU

### Workflow 3: Large-Scale Distributed Training

```bash
# 1. Test single GPU
python train.py --local

# 2. Test 2 GPUs (verify scaling)
# ... modify config for 2 GPUs ...

# 3. Scale to 4-8 GPUs
# ... use multi_gpu_4xa100 preset ...

# 4. Use preemptible
# ... enable checkpointing ...
```

**Time:** 2-4 hours setup + training
**Cost:** $15-60/hour

## Troubleshooting

### BigQuery ML

**"Insufficient permissions":**
```bash
gcloud projects add-iam-policy-binding PROJECT_ID \
  --member=user:EMAIL \
  --role=roles/bigquery.dataEditor
```

**"Model training failed":**
- Check for NULL values
- Verify data types
- Review TRANSFORM clause

### Vertex AI

**"Service account lacks permissions":**
```bash
gcloud projects add-iam-policy-binding PROJECT_ID \
  --member=serviceAccount:SA_EMAIL \
  --role=roles/aiplatform.user
```

**"GPU quota exceeded":**
- Request quota increase in console
- Try different region
- Use preemptible instances

**"Training job crashes":**
- Reduce batch size (OOM)
- Check CUDA compatibility
- Review Cloud Logging

## Resources

- **GCP ML Docs**: https://cloud.google.com/vertex-ai/docs
- **BigQuery ML**: https://cloud.google.com/bigquery-ml/docs
- **Pricing**: https://cloud.google.com/products/calculator
- **Vertex AI Samples**: https://github.com/GoogleCloudPlatform/vertex-ai-samples

## Integration with ML Training Plugin

This skill integrates with:

- **training-patterns**: Provides GCP configs for training scripts
- **cost-calculator**: Uses GCP pricing for budget planning
- **monitoring-dashboard**: Integrates with Vertex AI TensorBoard
- **validation-scripts**: Validates GCP credentials
- **integration-helpers**: Deploys to Vertex AI endpoints

## Support

For issues or questions:

1. Check troubleshooting section
2. Review GCP documentation
3. Check Cloud Logging for errors
4. Verify IAM permissions
5. Test with minimal example

## Version

- **Version**: 1.0.0
- **Last Updated**: 2025-01-04
- **Compatible with**: BigQuery ML (all versions), Vertex AI SDK 1.30+
