-- BigQuery ML Regression Example: NYC Taxi Trip Duration Prediction
-- Complete end-to-end example using public BigQuery dataset

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 1: Create Training Dataset
-- Sample NYC taxi data and engineer features
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE TABLE `your_project.ml_models.taxi_training_data` AS
SELECT
  -- Target variable: trip duration in minutes
  TIMESTAMP_DIFF(dropoff_datetime, pickup_datetime, MINUTE) AS trip_duration_minutes,

  -- Time features
  EXTRACT(HOUR FROM pickup_datetime) AS pickup_hour,
  EXTRACT(DAYOFWEEK FROM pickup_datetime) AS pickup_day_of_week,
  EXTRACT(MONTH FROM pickup_datetime) AS pickup_month,

  -- Location features
  pickup_latitude,
  pickup_longitude,
  dropoff_latitude,
  dropoff_longitude,

  -- Distance feature (approximate)
  ABS(pickup_latitude - dropoff_latitude) + ABS(pickup_longitude - dropoff_longitude) AS manhattan_distance,

  -- Trip info
  passenger_count,

  -- Create train/test split
  CASE
    WHEN RAND() < 0.8 THEN 'train'
    ELSE 'test'
  END AS data_split

FROM
  `bigquery-public-data.new_york_taxi_trips.tlc_yellow_trips_2015`
WHERE
  -- Filter for reasonable trips
  trip_distance > 0
  AND trip_distance < 100
  AND fare_amount > 0
  AND fare_amount < 500
  AND passenger_count > 0
  AND passenger_count < 9
  AND TIMESTAMP_DIFF(dropoff_datetime, pickup_datetime, MINUTE) > 0
  AND TIMESTAMP_DIFF(dropoff_datetime, pickup_datetime, MINUTE) < 180
  AND pickup_latitude BETWEEN 40.5 AND 41.0
  AND pickup_longitude BETWEEN -74.5 AND -73.5
  AND dropoff_latitude BETWEEN 40.5 AND 41.0
  AND dropoff_longitude BETWEEN -74.5 AND -73.5

-- Sample to make training faster (remove LIMIT for full dataset)
LIMIT 100000;

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 2: Train BOOSTED_TREE_REGRESSOR Model
-- XGBoost model for best accuracy
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `your_project.ml_models.taxi_duration_model`
OPTIONS(
  model_type='BOOSTED_TREE_REGRESSOR',
  input_label_cols=['trip_duration_minutes'],

  -- Boosting parameters
  max_iterations=50,
  learning_rate=0.3,
  subsample=0.8,
  max_tree_depth=6,
  min_tree_child_weight=1,

  -- Data split
  data_split_method='CUSTOM',
  data_split_col='data_split',

  -- Early stopping
  early_stop=TRUE,
  min_rel_progress=0.01
) AS
SELECT
  pickup_hour,
  pickup_day_of_week,
  pickup_month,
  pickup_latitude,
  pickup_longitude,
  dropoff_latitude,
  dropoff_longitude,
  manhattan_distance,
  passenger_count,
  trip_duration_minutes
FROM
  `your_project.ml_models.taxi_training_data`
WHERE
  data_split = 'train';

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 3: Evaluate Model Performance
-- ═══════════════════════════════════════════════════════════════════════════

-- Overall evaluation metrics
SELECT
  *
FROM
  ML.EVALUATE(
    MODEL `your_project.ml_models.taxi_duration_model`,
    (
      SELECT * FROM `your_project.ml_models.taxi_training_data`
      WHERE data_split = 'test'
    )
  );

-- Expected output: mean_absolute_error, mean_squared_error, r2_score, etc.

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 4: Feature Importance Analysis
-- ═══════════════════════════════════════════════════════════════════════════

SELECT
  feature,
  importance
FROM
  ML.FEATURE_IMPORTANCE(MODEL `your_project.ml_models.taxi_duration_model`)
ORDER BY
  importance DESC;

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 5: Detailed Predictions with Actual vs Predicted
-- ═══════════════════════════════════════════════════════════════════════════

SELECT
  predicted_trip_duration_minutes,
  trip_duration_minutes AS actual_trip_duration_minutes,
  ABS(predicted_trip_duration_minutes - trip_duration_minutes) AS error_minutes,
  pickup_hour,
  manhattan_distance,
  passenger_count
FROM
  ML.PREDICT(
    MODEL `your_project.ml_models.taxi_duration_model`,
    (
      SELECT * FROM `your_project.ml_models.taxi_training_data`
      WHERE data_split = 'test'
      LIMIT 100
    )
  )
ORDER BY
  error_minutes DESC;

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 6: Training Information and Convergence
-- ═══════════════════════════════════════════════════════════════════════════

SELECT
  iteration,
  loss,
  eval_loss,
  learning_rate
FROM
  ML.TRAINING_INFO(MODEL `your_project.ml_models.taxi_duration_model`)
ORDER BY
  iteration;

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 7: Make Predictions on New Data
-- Example prediction for a new trip
-- ═══════════════════════════════════════════════════════════════════════════

SELECT
  predicted_trip_duration_minutes
FROM
  ML.PREDICT(
    MODEL `your_project.ml_models.taxi_duration_model`,
    (
      SELECT
        15 AS pickup_hour,  -- 3 PM
        2 AS pickup_day_of_week,  -- Monday
        6 AS pickup_month,  -- June
        40.758896 AS pickup_latitude,  -- Times Square
        -73.985130 AS pickup_longitude,
        40.748817 AS dropoff_latitude,  -- Near Empire State
        -73.985428 AS dropoff_longitude,
        ABS(40.758896 - 40.748817) + ABS(-73.985130 - (-73.985428)) AS manhattan_distance,
        2 AS passenger_count
    )
  );

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 8: Model Performance by Hour of Day
-- Analyze how model performs at different times
-- ═══════════════════════════════════════════════════════════════════════════

SELECT
  pickup_hour,
  COUNT(*) AS num_predictions,
  AVG(ABS(predicted_trip_duration_minutes - trip_duration_minutes)) AS mean_absolute_error,
  STDDEV(ABS(predicted_trip_duration_minutes - trip_duration_minutes)) AS std_error
FROM
  ML.PREDICT(
    MODEL `your_project.ml_models.taxi_duration_model`,
    (
      SELECT * FROM `your_project.ml_models.taxi_training_data`
      WHERE data_split = 'test'
    )
  )
GROUP BY
  pickup_hour
ORDER BY
  pickup_hour;

-- ═══════════════════════════════════════════════════════════════════════════
-- STEP 9: Export Model (Optional)
-- Export for use outside BigQuery
-- ═══════════════════════════════════════════════════════════════════════════

-- Note: Requires gs:// bucket in same region as BigQuery dataset
EXPORT MODEL `your_project.ml_models.taxi_duration_model`
OPTIONS(URI='gs://your_bucket/models/taxi_duration_model');

-- ═══════════════════════════════════════════════════════════════════════════
-- RESULTS INTERPRETATION
-- ═══════════════════════════════════════════════════════════════════════════

/*
Expected Performance:
- Mean Absolute Error: 3-5 minutes
- R² Score: 0.7-0.8
- Most important features: manhattan_distance, pickup_hour

Good predictions for:
- Typical trips during regular hours
- Trips with moderate distances

Less accurate for:
- Rush hour traffic (model doesn't know traffic)
- Very long trips
- Unusual routes

Improvements:
1. Add traffic data (if available)
2. Add weather data
3. Include day of month (payday effects)
4. Add pickup/dropoff zone features
5. Include previous trip patterns
6. Use DNN for more complex patterns
7. Add time-based features (is_holiday, is_weekend)

Cost Estimate:
- Data processed: ~100,000 rows × 10 columns × 100 bytes = 0.0001 TB
- Training: ~50 iterations × 2 scans × 0.0001 TB = 0.01 TB
- Cost: 0.01 TB × $5/TB = $0.05

For full dataset (millions of rows):
- 1M rows = ~0.001 TB
- Training cost = ~$0.50
- Very cost-effective!
*/
