"""
Vertex AI Hugging Face Fine-tuning Example

Production-ready template for fine-tuning transformer models on Vertex AI.
Uses Hugging Face Trainer API for simplified training.

Features:
- Hugging Face Trainer integration
- Automatic mixed precision
- Built-in evaluation and metrics
- Hyperparameter tuning with Vertex AI
- Model versioning
- Endpoint deployment

Submit to Vertex AI:
    gcloud ai custom-jobs create \
      --region=us-central1 \
      --display-name=hf-finetuning \
      --config=gpu_config.yaml
"""

import os
import argparse
import logging
from dataclasses import dataclass, field
from typing import Optional

import torch
from datasets import load_dataset
from transformers import (
    AutoTokenizer,
    AutoModelForSequenceClassification,
    TrainingArguments,
    Trainer,
    EarlyStoppingCallback,
    HfArgumentParser
)
from google.cloud import storage, aiplatform
import numpy as np
from sklearn.metrics import accuracy_score, precision_recall_fscore_support

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


@dataclass
class ModelArguments:
    """Arguments for model configuration"""
    model_name_or_path: str = field(
        default="distilbert-base-uncased",
        metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    num_labels: int = field(
        default=2,
        metadata={"help": "Number of classification labels"}
    )


@dataclass
class DataArguments:
    """Arguments for data configuration"""
    dataset_name: str = field(
        default="imdb",
        metadata={"help": "HuggingFace dataset name"}
    )
    dataset_config: Optional[str] = field(
        default=None,
        metadata={"help": "Dataset configuration name"}
    )
    max_length: int = field(
        default=512,
        metadata={"help": "Maximum sequence length"}
    )
    text_column: str = field(
        default="text",
        metadata={"help": "Name of text column"}
    )
    label_column: str = field(
        default="label",
        metadata={"help": "Name of label column"}
    )


@dataclass
class GCPArguments:
    """Arguments for GCP configuration"""
    project_id: str = field(
        default="your_project_id_here",
        metadata={"help": "GCP project ID"}
    )
    region: str = field(
        default="us-central1",
        metadata={"help": "GCP region"}
    )
    bucket_name: str = field(
        default="your_bucket_here",
        metadata={"help": "GCS bucket name"}
    )
    model_dir: str = field(
        default="models",
        metadata={"help": "Model directory in GCS bucket"}
    )
    deploy_endpoint: bool = field(
        default=False,
        metadata={"help": "Deploy model to Vertex AI endpoint after training"}
    )


def compute_metrics(pred):
    """Compute metrics for evaluation"""
    labels = pred.label_ids
    preds = pred.predictions.argmax(-1)

    # Calculate metrics
    accuracy = accuracy_score(labels, preds)
    precision, recall, f1, _ = precision_recall_fscore_support(labels, preds, average='weighted')

    return {
        'accuracy': accuracy,
        'precision': precision,
        'recall': recall,
        'f1': f1
    }


class CustomTrainer(Trainer):
    """Custom trainer with GCS checkpointing"""

    def __init__(self, *args, gcp_args=None, **kwargs):
        super().__init__(*args, **kwargs)
        self.gcp_args = gcp_args

        if gcp_args:
            self.storage_client = storage.Client()
        else:
            self.storage_client = None

    def _save_checkpoint(self, model, trial, metrics=None):
        """Override to save checkpoints to GCS"""
        # First save locally
        checkpoint_folder = super()._save_checkpoint(model, trial, metrics)

        # Then upload to GCS if configured
        if self.storage_client and self.gcp_args:
            self._upload_to_gcs(checkpoint_folder)

        return checkpoint_folder

    def _upload_to_gcs(self, local_path):
        """Upload checkpoint to GCS"""
        try:
            bucket = self.storage_client.bucket(self.gcp_args.bucket_name)

            for root, dirs, files in os.walk(local_path):
                for file in files:
                    local_file = os.path.join(root, file)
                    relative_path = os.path.relpath(local_file, local_path)
                    gcs_path = f"{self.gcp_args.model_dir}/checkpoints/{os.path.basename(local_path)}/{relative_path}"

                    blob = bucket.blob(gcs_path)
                    blob.upload_from_filename(local_file)

            logger.info(f"Uploaded checkpoint to gs://{self.gcp_args.bucket_name}/{gcs_path}")
        except Exception as e:
            logger.error(f"Failed to upload to GCS: {e}")


def prepare_dataset(data_args, tokenizer):
    """Load and prepare dataset"""
    logger.info(f"Loading dataset: {data_args.dataset_name}")

    # Load dataset from HuggingFace
    if data_args.dataset_config:
        dataset = load_dataset(data_args.dataset_name, data_args.dataset_config)
    else:
        dataset = load_dataset(data_args.dataset_name)

    # Tokenization function
    def tokenize_function(examples):
        return tokenizer(
            examples[data_args.text_column],
            padding='max_length',
            truncation=True,
            max_length=data_args.max_length
        )

    # Tokenize datasets
    logger.info("Tokenizing datasets...")
    tokenized_datasets = dataset.map(
        tokenize_function,
        batched=True,
        remove_columns=[col for col in dataset['train'].column_names if col != data_args.label_column]
    )

    # Rename label column to 'labels' (required by Trainer)
    if data_args.label_column != 'labels':
        tokenized_datasets = tokenized_datasets.rename_column(data_args.label_column, 'labels')

    # Format for PyTorch
    tokenized_datasets.set_format('torch')

    return tokenized_datasets['train'], tokenized_datasets.get('test', tokenized_datasets.get('validation'))


def upload_final_model(model, tokenizer, gcp_args):
    """Upload final model to GCS"""
    logger.info("Uploading final model to GCS...")

    # Save locally
    local_model_dir = '/tmp/final_model'
    model.save_pretrained(local_model_dir)
    tokenizer.save_pretrained(local_model_dir)

    # Upload to GCS
    storage_client = storage.Client()
    bucket = storage_client.bucket(gcp_args.bucket_name)

    for root, dirs, files in os.walk(local_model_dir):
        for file in files:
            local_file = os.path.join(root, file)
            relative_path = os.path.relpath(local_file, local_model_dir)
            gcs_path = f"{gcp_args.model_dir}/final_model/{relative_path}"

            blob = bucket.blob(gcs_path)
            blob.upload_from_filename(local_file)

    logger.info(f"Model uploaded to gs://{gcp_args.bucket_name}/{gcp_args.model_dir}/final_model")


def deploy_to_endpoint(model_path, gcp_args):
    """Deploy model to Vertex AI endpoint"""
    logger.info("Deploying model to Vertex AI endpoint...")

    # Initialize Vertex AI
    aiplatform.init(project=gcp_args.project_id, location=gcp_args.region)

    # Upload model to Model Registry
    model = aiplatform.Model.upload(
        display_name='sentiment-classifier',
        artifact_uri=f"gs://{gcp_args.bucket_name}/{gcp_args.model_dir}/final_model",
        serving_container_image_uri='us-docker.pkg.dev/vertex-ai/prediction/pytorch-gpu.1-13:latest',
    )

    logger.info(f"Model uploaded: {model.resource_name}")

    # Create endpoint
    endpoint = aiplatform.Endpoint.create(
        display_name='sentiment-classifier-endpoint'
    )

    logger.info(f"Endpoint created: {endpoint.resource_name}")

    # Deploy model to endpoint
    model.deploy(
        endpoint=endpoint,
        deployed_model_display_name='sentiment-classifier-v1',
        machine_type='n1-standard-4',
        accelerator_type='NVIDIA_TESLA_T4',
        accelerator_count=1,
        traffic_percentage=100,
        min_replica_count=1,
        max_replica_count=3
    )

    logger.info(f"Model deployed to endpoint: {endpoint.resource_name}")
    return endpoint


def main():
    # Parse arguments
    parser = HfArgumentParser((ModelArguments, DataArguments, TrainingArguments, GCPArguments))
    model_args, data_args, training_args, gcp_args = parser.parse_args_into_dataclasses()

    # Log GPU info
    if torch.cuda.is_available():
        logger.info(f"Training on GPU: {torch.cuda.get_device_name(0)}")
        logger.info(f"GPU Memory: {torch.cuda.get_device_properties(0).total_memory / 1e9:.2f} GB")
    else:
        logger.info("Training on CPU")

    # Load tokenizer
    logger.info(f"Loading tokenizer: {model_args.model_name_or_path}")
    tokenizer = AutoTokenizer.from_pretrained(model_args.model_name_or_path)

    # Prepare datasets
    train_dataset, eval_dataset = prepare_dataset(data_args, tokenizer)

    logger.info(f"Train examples: {len(train_dataset)}")
    logger.info(f"Eval examples: {len(eval_dataset) if eval_dataset else 0}")

    # Load model
    logger.info(f"Loading model: {model_args.model_name_or_path}")
    model = AutoModelForSequenceClassification.from_pretrained(
        model_args.model_name_or_path,
        num_labels=model_args.num_labels
    )

    # Create Trainer
    trainer = CustomTrainer(
        model=model,
        args=training_args,
        train_dataset=train_dataset,
        eval_dataset=eval_dataset,
        tokenizer=tokenizer,
        compute_metrics=compute_metrics,
        callbacks=[EarlyStoppingCallback(early_stopping_patience=3)],
        gcp_args=gcp_args
    )

    # Train
    logger.info("Starting training...")
    train_result = trainer.train()

    # Log metrics
    logger.info(f"Training metrics: {train_result.metrics}")

    # Evaluate
    if eval_dataset:
        logger.info("Evaluating model...")
        eval_metrics = trainer.evaluate()
        logger.info(f"Evaluation metrics: {eval_metrics}")

    # Save final model locally
    logger.info("Saving final model...")
    trainer.save_model(training_args.output_dir)

    # Upload to GCS
    if gcp_args.bucket_name != "your_bucket_here":
        upload_final_model(model, tokenizer, gcp_args)

        # Deploy to endpoint if requested
        if gcp_args.deploy_endpoint:
            deploy_to_endpoint(
                f"gs://{gcp_args.bucket_name}/{gcp_args.model_dir}/final_model",
                gcp_args
            )

    logger.info("Training complete!")


if __name__ == '__main__':
    # Example usage with default arguments
    import sys

    if len(sys.argv) == 1:
        # Add default arguments for testing
        sys.argv.extend([
            '--output_dir', './output',
            '--num_train_epochs', '3',
            '--per_device_train_batch_size', '16',
            '--per_device_eval_batch_size', '16',
            '--learning_rate', '2e-5',
            '--weight_decay', '0.01',
            '--warmup_ratio', '0.1',
            '--logging_steps', '100',
            '--evaluation_strategy', 'epoch',
            '--save_strategy', 'epoch',
            '--load_best_model_at_end', 'True',
            '--metric_for_best_model', 'f1',
            '--fp16', 'True',  # Mixed precision
            '--report_to', 'none',  # Disable wandb/tensorboard for simplicity
        ])

    main()
