#!/bin/bash
# Configure GCP authentication for ML training workflows

set -e

echo "=== GCP Authentication Configuration ==="
echo ""

# Check if gcloud is installed
if ! command -v gcloud &> /dev/null; then
    echo "Error: gcloud CLI not found. Please install:"
    echo "https://cloud.google.com/sdk/docs/install"
    exit 1
fi

# Select authentication method
echo "Select authentication method:"
echo "1) User Account (gcloud auth) - For local development"
echo "2) Service Account Key File - For automation/CI/CD"
echo "3) Workload Identity - For GKE deployments"
echo "4) Application Default Credentials - Auto-detect"
read -p "Enter choice (1-4): " AUTH_CHOICE

case $AUTH_CHOICE in
    1) AUTH_METHOD="user_account" ;;
    2) AUTH_METHOD="service_account" ;;
    3) AUTH_METHOD="workload_identity" ;;
    4) AUTH_METHOD="application_default" ;;
    *) AUTH_METHOD="user_account"; echo "Using default: User Account" ;;
esac

# Get project ID
echo ""
echo "Enter your GCP Project ID:"
read -r PROJECT_ID

if [ -z "$PROJECT_ID" ]; then
    echo "Error: Project ID is required"
    exit 1
fi

gcloud config set project "$PROJECT_ID"

# Handle authentication based on method
if [ "$AUTH_METHOD" = "user_account" ]; then
    echo ""
    echo "Authenticating with user account..."

    # Check if already authenticated
    if gcloud auth list --filter=status:ACTIVE --format="value(account)" &> /dev/null; then
        CURRENT_ACCOUNT=$(gcloud auth list --filter=status:ACTIVE --format="value(account)")
        echo "Already authenticated as: $CURRENT_ACCOUNT"
        read -p "Re-authenticate? (y/n): " REAUTH
        if [ "$REAUTH" = "y" ]; then
            gcloud auth login
        fi
    else
        gcloud auth login
    fi

    # Set application default credentials
    echo ""
    echo "Setting application default credentials..."
    gcloud auth application-default login

    SERVICE_ACCOUNT_EMAIL="user-account"
    KEY_FILE_PATH="~/.config/gcloud/application_default_credentials.json"

elif [ "$AUTH_METHOD" = "service_account" ]; then
    echo ""
    echo "Service Account Setup"
    echo ""

    # Check for existing service account
    echo "Do you have an existing service account key file? (y/n)"
    read -r HAS_KEY

    if [ "$HAS_KEY" = "y" ]; then
        echo "Enter path to service account key file:"
        read -r KEY_FILE_PATH

        if [ ! -f "$KEY_FILE_PATH" ]; then
            echo "Error: Key file not found at $KEY_FILE_PATH"
            exit 1
        fi

        # Extract service account email from key file
        SERVICE_ACCOUNT_EMAIL=$(grep -o '"client_email": "[^"]*"' "$KEY_FILE_PATH" | cut -d'"' -f4)

    else
        echo "Creating new service account..."
        echo ""

        echo "Enter service account name (e.g., ml-training-sa):"
        read -r SA_NAME

        if [ -z "$SA_NAME" ]; then
            SA_NAME="ml-training-sa"
            echo "Using default: $SA_NAME"
        fi

        SERVICE_ACCOUNT_EMAIL="${SA_NAME}@${PROJECT_ID}.iam.gserviceaccount.com"

        # Create service account
        gcloud iam service-accounts create "$SA_NAME" \
            --display-name="ML Training Service Account" \
            --project="$PROJECT_ID"

        echo "✓ Created service account: $SERVICE_ACCOUNT_EMAIL"

        # Grant necessary roles
        echo ""
        echo "Granting IAM roles..."

        # BigQuery roles
        gcloud projects add-iam-policy-binding "$PROJECT_ID" \
            --member="serviceAccount:$SERVICE_ACCOUNT_EMAIL" \
            --role="roles/bigquery.dataEditor" \
            --condition=None

        gcloud projects add-iam-policy-binding "$PROJECT_ID" \
            --member="serviceAccount:$SERVICE_ACCOUNT_EMAIL" \
            --role="roles/bigquery.jobUser" \
            --condition=None

        # Vertex AI roles
        gcloud projects add-iam-policy-binding "$PROJECT_ID" \
            --member="serviceAccount:$SERVICE_ACCOUNT_EMAIL" \
            --role="roles/aiplatform.user" \
            --condition=None

        # Storage roles
        gcloud projects add-iam-policy-binding "$PROJECT_ID" \
            --member="serviceAccount:$SERVICE_ACCOUNT_EMAIL" \
            --role="roles/storage.objectAdmin" \
            --condition=None

        echo "✓ Granted IAM roles"

        # Create and download key
        echo ""
        KEY_FILE_PATH="./gcp-ml-training-key.json"

        gcloud iam service-accounts keys create "$KEY_FILE_PATH" \
            --iam-account="$SERVICE_ACCOUNT_EMAIL" \
            --project="$PROJECT_ID"

        echo "✓ Created key file: $KEY_FILE_PATH"
        echo ""
        echo "⚠ SECURITY WARNING:"
        echo "  - This key file grants access to your GCP project"
        echo "  - Never commit this file to version control"
        echo "  - Store securely (Secret Manager, vault, etc.)"
        echo "  - Rotate keys regularly (90 days recommended)"
    fi

    # Set GOOGLE_APPLICATION_CREDENTIALS
    export GOOGLE_APPLICATION_CREDENTIALS="$KEY_FILE_PATH"

    # Activate service account
    gcloud auth activate-service-account "$SERVICE_ACCOUNT_EMAIL" --key-file="$KEY_FILE_PATH"

elif [ "$AUTH_METHOD" = "workload_identity" ]; then
    echo ""
    echo "Workload Identity Setup"
    echo ""
    echo "This requires a GKE cluster with Workload Identity enabled."
    echo ""

    echo "Enter Kubernetes service account name:"
    read -r K8S_SA_NAME

    echo "Enter Kubernetes namespace:"
    read -r K8S_NAMESPACE

    echo "Enter GCP service account name (e.g., ml-training-sa):"
    read -r GCP_SA_NAME

    SERVICE_ACCOUNT_EMAIL="${GCP_SA_NAME}@${PROJECT_ID}.iam.gserviceaccount.com"

    # Create GCP service account if doesn't exist
    if ! gcloud iam service-accounts describe "$SERVICE_ACCOUNT_EMAIL" &> /dev/null; then
        echo "Creating GCP service account..."
        gcloud iam service-accounts create "$GCP_SA_NAME" \
            --display-name="ML Training Service Account" \
            --project="$PROJECT_ID"
    fi

    # Grant IAM roles (same as service account method)
    echo "Granting IAM roles..."
    gcloud projects add-iam-policy-binding "$PROJECT_ID" \
        --member="serviceAccount:$SERVICE_ACCOUNT_EMAIL" \
        --role="roles/aiplatform.user" \
        --condition=None

    gcloud projects add-iam-policy-binding "$PROJECT_ID" \
        --member="serviceAccount:$SERVICE_ACCOUNT_EMAIL" \
        --role="roles/storage.objectAdmin" \
        --condition=None

    # Bind Kubernetes SA to GCP SA
    echo "Binding Kubernetes service account to GCP service account..."
    gcloud iam service-accounts add-iam-policy-binding "$SERVICE_ACCOUNT_EMAIL" \
        --role="roles/iam.workloadIdentityUser" \
        --member="serviceAccount:${PROJECT_ID}.svc.id.goog[${K8S_NAMESPACE}/${K8S_SA_NAME}]"

    echo "✓ Workload Identity configured"
    echo ""
    echo "Add this annotation to your Kubernetes service account:"
    echo "  iam.gke.io/gcp-service-account: $SERVICE_ACCOUNT_EMAIL"

    KEY_FILE_PATH="workload-identity"

else # application_default
    echo ""
    echo "Using Application Default Credentials..."
    echo "This will auto-detect credentials from environment."

    gcloud auth application-default login

    SERVICE_ACCOUNT_EMAIL="application-default"
    KEY_FILE_PATH="~/.config/gcloud/application_default_credentials.json"
fi

# Create configuration file
cat > .gcp_auth_config << EOF
# GCP Authentication Configuration
# Generated on $(date)

export GCP_PROJECT_ID="$PROJECT_ID"
export GCP_AUTH_METHOD="$AUTH_METHOD"
export GCP_SERVICE_ACCOUNT="$SERVICE_ACCOUNT_EMAIL"
EOF

if [ "$AUTH_METHOD" = "service_account" ]; then
    echo "export GOOGLE_APPLICATION_CREDENTIALS=\"$KEY_FILE_PATH\"" >> .gcp_auth_config
fi

echo "✓ Created .gcp_auth_config"

# Add to .gitignore
if [ -f .gitignore ]; then
    if ! grep -q "gcp-ml-training-key.json" .gitignore; then
        echo "" >> .gitignore
        echo "# GCP credentials" >> .gitignore
        echo "gcp-ml-training-key.json" >> .gitignore
        echo "*.json" >> .gitignore
        echo ".gcp_auth_config" >> .gitignore
        echo "✓ Added credentials to .gitignore"
    fi
else
    cat > .gitignore << EOF
# GCP credentials
gcp-ml-training-key.json
*.json
.gcp_auth_config
EOF
    echo "✓ Created .gitignore"
fi

# Validate permissions
echo ""
echo "Validating permissions..."

# Test BigQuery access
echo -n "BigQuery access... "
if gcloud projects describe "$PROJECT_ID" &> /dev/null; then
    echo "✓"
else
    echo "✗ (May need bigquery.dataEditor and bigquery.jobUser roles)"
fi

# Test Vertex AI access
echo -n "Vertex AI access... "
if gcloud ai-platform models list --region=us-central1 &> /dev/null 2>&1 || \
   gcloud ai custom-jobs list --region=us-central1 &> /dev/null 2>&1; then
    echo "✓"
else
    echo "✗ (May need aiplatform.user role)"
fi

# Test Storage access
echo -n "Cloud Storage access... "
if gsutil ls "gs://" &> /dev/null; then
    echo "✓"
else
    echo "✗ (May need storage.objectAdmin role)"
fi

# Summary
echo ""
echo "════════════════════════════════════════"
echo "✓ Authentication Configuration Complete!"
echo "════════════════════════════════════════"
echo ""
echo "Configuration:"
echo "  Project ID: $PROJECT_ID"
echo "  Auth Method: $AUTH_METHOD"
echo "  Service Account: $SERVICE_ACCOUNT_EMAIL"
if [ "$AUTH_METHOD" = "service_account" ]; then
    echo "  Key File: $KEY_FILE_PATH"
fi
echo ""
echo "Files Created:"
echo "  • .gcp_auth_config - Environment configuration"
if [ "$AUTH_METHOD" = "service_account" ] && [ "$HAS_KEY" != "y" ]; then
    echo "  • $KEY_FILE_PATH - Service account key (KEEP SECURE!)"
fi
echo "  • .gitignore - Protects credentials from git"
echo ""
echo "To use in your shell:"
echo "  source .gcp_auth_config"
echo ""
echo "To use in Python:"
if [ "$AUTH_METHOD" = "service_account" ]; then
    echo "  import os"
    echo "  os.environ['GOOGLE_APPLICATION_CREDENTIALS'] = '$KEY_FILE_PATH'"
else
    echo "  # Authentication is automatically detected"
    echo "  from google.cloud import bigquery, aiplatform"
fi
echo ""
echo "Security Best Practices:"
echo "  ✓ Never commit key files to git"
echo "  ✓ Rotate service account keys every 90 days"
echo "  ✓ Use Workload Identity for GKE workloads"
echo "  ✓ Grant minimum necessary permissions (least privilege)"
echo "  ✓ Enable Cloud Audit Logs for security monitoring"
echo ""
