#!/bin/bash
# Estimate GCP ML training costs for BigQuery ML and Vertex AI

set -e

echo "=== GCP ML Training Cost Estimator ==="
echo ""

# Platform selection
echo "Select platform:"
echo "1) BigQuery ML"
echo "2) Vertex AI - GPU Training"
echo "3) Vertex AI - TPU Training"
echo "4) Vertex AI - CPU Training"
read -p "Enter choice (1-4): " PLATFORM_CHOICE

case $PLATFORM_CHOICE in
    1) PLATFORM="bigquery_ml" ;;
    2) PLATFORM="vertex_gpu" ;;
    3) PLATFORM="vertex_tpu" ;;
    4) PLATFORM="vertex_cpu" ;;
    *) PLATFORM="bigquery_ml"; echo "Using default: BigQuery ML" ;;
esac

# BigQuery ML cost estimation
if [ "$PLATFORM" = "bigquery_ml" ]; then
    echo ""
    echo "=== BigQuery ML Cost Estimation ==="
    echo ""

    echo "Select model type:"
    echo "1) LINEAR_REG / LOGISTIC_REG - Standard SQL processing"
    echo "2) BOOSTED_TREE - XGBoost models"
    echo "3) DNN (Deep Neural Network) - Neural networks"
    echo "4) AUTOML - AutoML Tables"
    read -p "Enter choice (1-4): " MODEL_CHOICE

    case $MODEL_CHOICE in
        1) MODEL_TYPE="linear" ;;
        2) MODEL_TYPE="boosted_tree" ;;
        3) MODEL_TYPE="dnn" ;;
        4) MODEL_TYPE="automl" ;;
        *) MODEL_TYPE="linear" ;;
    esac

    echo ""
    echo "Estimate training data size:"
    read -p "Number of rows (millions): " ROWS_MILLIONS
    read -p "Number of columns: " COLUMNS
    read -p "Average bytes per row (default: 100): " BYTES_PER_ROW
    BYTES_PER_ROW=${BYTES_PER_ROW:-100}

    # Calculate data size in TB
    ROWS=$(echo "$ROWS_MILLIONS * 1000000" | bc)
    TOTAL_BYTES=$(echo "$ROWS * $COLUMNS * $BYTES_PER_ROW" | bc)
    DATA_SIZE_TB=$(echo "scale=4; $TOTAL_BYTES / 1099511627776" | bc)

    echo ""
    echo "Training iterations:"
    read -p "Number of training iterations (default: 20): " ITERATIONS
    ITERATIONS=${ITERATIONS:-20}

    # Calculate costs
    if [ "$MODEL_TYPE" = "automl" ]; then
        # AutoML has per-hour pricing
        echo ""
        read -p "Estimated training time in hours (default: 2): " TRAINING_HOURS
        TRAINING_HOURS=${TRAINING_HOURS:-2}

        COMPUTE_COST=$(echo "$TRAINING_HOURS * 19.32" | bc)
        DATA_COST=0
    else
        # Standard BigQuery ML - charged per TB processed
        PRICE_PER_TB=5.00
        SCANS_PER_ITERATION=2  # Training typically scans data ~2x per iteration

        TOTAL_TB_PROCESSED=$(echo "scale=4; $DATA_SIZE_TB * $SCANS_PER_ITERATION * $ITERATIONS" | bc)
        COMPUTE_COST=$(echo "scale=2; $TOTAL_TB_PROCESSED * $PRICE_PER_TB" | bc)
        DATA_COST=0  # No separate storage cost for active queries
    fi

    # Storage cost (if persisting models)
    echo ""
    read -p "Store model for predictions? (y/n): " STORE_MODEL
    if [ "$STORE_MODEL" = "y" ]; then
        read -p "Expected model size in GB (default: 1): " MODEL_SIZE_GB
        MODEL_SIZE_GB=${MODEL_SIZE_GB:-1}

        STORAGE_COST_PER_GB_MONTH=0.02
        STORAGE_COST=$(echo "scale=2; $MODEL_SIZE_GB * $STORAGE_COST_PER_GB_MONTH" | bc)
    else
        STORAGE_COST=0
    fi

    # Prediction costs
    echo ""
    read -p "Estimate prediction costs? (y/n): " PREDICT
    if [ "$PREDICT" = "y" ]; then
        read -p "Prediction data size in TB (default: 0.1): " PREDICT_TB
        PREDICT_TB=${PREDICT_TB:-0.1}

        PREDICT_COST=$(echo "scale=2; $PREDICT_TB * $PRICE_PER_TB" | bc)
    else
        PREDICT_COST=0
    fi

    TOTAL_COST=$(echo "scale=2; $COMPUTE_COST + $DATA_COST + $STORAGE_COST + $PREDICT_COST" | bc)

    # Display results
    echo ""
    echo "════════════════════════════════════════"
    echo "BigQuery ML Cost Estimate"
    echo "════════════════════════════════════════"
    echo ""
    echo "Model Type: $MODEL_TYPE"
    if [ "$MODEL_TYPE" = "automl" ]; then
        echo "Training Time: ${TRAINING_HOURS}h"
        echo ""
        echo "Cost Breakdown:"
        echo "  Training: \$${COMPUTE_COST}"
    else
        echo "Data Size: ${DATA_SIZE_TB} TB"
        echo "Iterations: $ITERATIONS"
        echo "Total Data Processed: ${TOTAL_TB_PROCESSED} TB"
        echo ""
        echo "Cost Breakdown:"
        echo "  Training: \$${COMPUTE_COST} (${TOTAL_TB_PROCESSED} TB × \$5/TB)"
    fi
    if (( $(echo "$STORAGE_COST > 0" | bc -l) )); then
        echo "  Storage: \$${STORAGE_COST}/month"
    fi
    if (( $(echo "$PREDICT_COST > 0" | bc -l) )); then
        echo "  Predictions: \$${PREDICT_COST}"
    fi
    echo ""
    echo "Total Estimated Cost: \$${TOTAL_COST}"
    echo ""

# Vertex AI GPU cost estimation
elif [ "$PLATFORM" = "vertex_gpu" ]; then
    echo ""
    echo "=== Vertex AI GPU Training Cost Estimation ==="
    echo ""

    echo "Select GPU type:"
    echo "1) NVIDIA_TESLA_T4 (16GB) - \$0.35/hr per GPU"
    echo "2) NVIDIA_TESLA_L4 (24GB) - \$0.66/hr per GPU"
    echo "3) NVIDIA_TESLA_V100 (16GB) - \$2.48/hr per GPU"
    echo "4) NVIDIA_TESLA_A100 (40GB) - \$3.67/hr per GPU"
    echo "5) NVIDIA_A100_80GB (80GB) - \$4.95/hr per GPU"
    read -p "Enter choice (1-5): " GPU_CHOICE

    case $GPU_CHOICE in
        1) GPU_NAME="T4"; GPU_PRICE=0.35 ;;
        2) GPU_NAME="L4"; GPU_PRICE=0.66 ;;
        3) GPU_NAME="V100"; GPU_PRICE=2.48 ;;
        4) GPU_NAME="A100 40GB"; GPU_PRICE=3.67 ;;
        5) GPU_NAME="A100 80GB"; GPU_PRICE=4.95 ;;
        *) GPU_NAME="T4"; GPU_PRICE=0.35 ;;
    esac

    read -p "Number of GPUs: " GPU_COUNT
    read -p "Training duration in hours: " TRAINING_HOURS

    # Base machine cost (n1-standard-8 is typical for GPU training)
    MACHINE_PRICE=0.38

    # Calculate costs
    GPU_COST=$(echo "scale=2; $GPU_PRICE * $GPU_COUNT * $TRAINING_HOURS" | bc)
    MACHINE_COST=$(echo "scale=2; $MACHINE_PRICE * $TRAINING_HOURS" | bc)
    COMPUTE_COST=$(echo "scale=2; $GPU_COST + $MACHINE_COST" | bc)

    # Storage costs
    echo ""
    echo "Storage estimation:"
    read -p "Dataset size in GB: " DATASET_GB
    read -p "Model checkpoint size in GB (default: 5): " CHECKPOINT_GB
    CHECKPOINT_GB=${CHECKPOINT_GB:-5}

    STORAGE_GB=$(echo "$DATASET_GB + $CHECKPOINT_GB * 3" | bc)  # ~3 checkpoints
    STORAGE_COST=$(echo "scale=2; $STORAGE_GB * 0.02" | bc)  # $0.02/GB/month

    # Preemptible discount
    echo ""
    read -p "Use preemptible instances? (60% discount, can be interrupted) (y/n): " PREEMPTIBLE
    if [ "$PREEMPTIBLE" = "y" ]; then
        COMPUTE_COST=$(echo "scale=2; $COMPUTE_COST * 0.4" | bc)  # 60% discount
        PREEMPTIBLE_NOTE=" (preemptible)"
    else
        PREEMPTIBLE_NOTE=""
    fi

    TOTAL_COST=$(echo "scale=2; $COMPUTE_COST + $STORAGE_COST" | bc)

    # Display results
    echo ""
    echo "════════════════════════════════════════"
    echo "Vertex AI GPU Cost Estimate"
    echo "════════════════════════════════════════"
    echo ""
    echo "Configuration:"
    echo "  GPU: ${GPU_COUNT}x $GPU_NAME"
    echo "  Duration: ${TRAINING_HOURS}h"
    echo ""
    echo "Cost Breakdown:"
    echo "  Compute: \$${COMPUTE_COST}${PREEMPTIBLE_NOTE}"
    echo "    - GPU: \$${GPU_COST} (${GPU_COUNT} × \$${GPU_PRICE}/hr × ${TRAINING_HOURS}h)"
    echo "    - Machine: \$${MACHINE_COST}"
    echo "  Storage: \$${STORAGE_COST}/month (${STORAGE_GB}GB)"
    echo ""
    echo "Total Estimated Cost: \$${TOTAL_COST}"
    echo ""

# Vertex AI TPU cost estimation
elif [ "$PLATFORM" = "vertex_tpu" ]; then
    echo ""
    echo "=== Vertex AI TPU Training Cost Estimation ==="
    echo ""

    echo "Select TPU type:"
    echo "1) TPU v2 (8 cores) - \$4.50/hr"
    echo "2) TPU v3 (8 cores) - \$8.00/hr"
    echo "3) TPU v4 (8 cores) - \$11.00/hr"
    echo "4) TPU v5e (8 cores) - \$2.50/hr"
    echo "5) TPU v3-32 (32 cores) - \$32.00/hr"
    echo "6) TPU v4-128 (128 cores) - \$176.00/hr"
    read -p "Enter choice (1-6): " TPU_CHOICE

    case $TPU_CHOICE in
        1) TPU_NAME="TPU v2-8"; TPU_PRICE=4.50 ;;
        2) TPU_NAME="TPU v3-8"; TPU_PRICE=8.00 ;;
        3) TPU_NAME="TPU v4-8"; TPU_PRICE=11.00 ;;
        4) TPU_NAME="TPU v5e-8"; TPU_PRICE=2.50 ;;
        5) TPU_NAME="TPU v3-32"; TPU_PRICE=32.00 ;;
        6) TPU_NAME="TPU v4-128"; TPU_PRICE=176.00 ;;
        *) TPU_NAME="TPU v3-8"; TPU_PRICE=8.00 ;;
    esac

    read -p "Training duration in hours: " TRAINING_HOURS

    # Calculate costs
    COMPUTE_COST=$(echo "scale=2; $TPU_PRICE * $TRAINING_HOURS" | bc)

    # Storage costs
    echo ""
    echo "Storage estimation:"
    read -p "Dataset size in GB: " DATASET_GB
    read -p "Model checkpoint size in GB (default: 10): " CHECKPOINT_GB
    CHECKPOINT_GB=${CHECKPOINT_GB:-10}

    STORAGE_GB=$(echo "$DATASET_GB + $CHECKPOINT_GB * 3" | bc)
    STORAGE_COST=$(echo "scale=2; $STORAGE_GB * 0.02" | bc)

    # Preemptible discount
    echo ""
    read -p "Use preemptible TPUs? (60% discount, can be interrupted) (y/n): " PREEMPTIBLE
    if [ "$PREEMPTIBLE" = "y" ]; then
        COMPUTE_COST=$(echo "scale=2; $COMPUTE_COST * 0.4" | bc)
        PREEMPTIBLE_NOTE=" (preemptible)"
    else
        PREEMPTIBLE_NOTE=""
    fi

    TOTAL_COST=$(echo "scale=2; $COMPUTE_COST + $STORAGE_COST" | bc)

    # Display results
    echo ""
    echo "════════════════════════════════════════"
    echo "Vertex AI TPU Cost Estimate"
    echo "════════════════════════════════════════"
    echo ""
    echo "Configuration:"
    echo "  TPU: $TPU_NAME"
    echo "  Duration: ${TRAINING_HOURS}h"
    echo ""
    echo "Cost Breakdown:"
    echo "  Compute: \$${COMPUTE_COST}${PREEMPTIBLE_NOTE}"
    echo "  Storage: \$${STORAGE_COST}/month (${STORAGE_GB}GB)"
    echo ""
    echo "Total Estimated Cost: \$${TOTAL_COST}"
    echo ""

# Vertex AI CPU cost estimation
else
    echo ""
    echo "=== Vertex AI CPU Training Cost Estimation ==="
    echo ""

    echo "Select machine type:"
    echo "1) n1-standard-4 (4 vCPU, 15GB) - \$0.19/hr"
    echo "2) n1-standard-8 (8 vCPU, 30GB) - \$0.38/hr"
    echo "3) n1-highmem-8 (8 vCPU, 52GB) - \$0.47/hr"
    echo "4) n1-highcpu-16 (16 vCPU, 14.4GB) - \$0.57/hr"
    read -p "Enter choice (1-4): " MACHINE_CHOICE

    case $MACHINE_CHOICE in
        1) MACHINE_NAME="n1-standard-4"; MACHINE_PRICE=0.19 ;;
        2) MACHINE_NAME="n1-standard-8"; MACHINE_PRICE=0.38 ;;
        3) MACHINE_NAME="n1-highmem-8"; MACHINE_PRICE=0.47 ;;
        4) MACHINE_NAME="n1-highcpu-16"; MACHINE_PRICE=0.57 ;;
        *) MACHINE_NAME="n1-standard-4"; MACHINE_PRICE=0.19 ;;
    esac

    read -p "Training duration in hours: " TRAINING_HOURS

    COMPUTE_COST=$(echo "scale=2; $MACHINE_PRICE * $TRAINING_HOURS" | bc)

    # Storage
    echo ""
    read -p "Dataset size in GB: " DATASET_GB
    STORAGE_COST=$(echo "scale=2; $DATASET_GB * 0.02" | bc)

    # Preemptible
    echo ""
    read -p "Use preemptible instances? (60% discount) (y/n): " PREEMPTIBLE
    if [ "$PREEMPTIBLE" = "y" ]; then
        COMPUTE_COST=$(echo "scale=2; $COMPUTE_COST * 0.4" | bc)
        PREEMPTIBLE_NOTE=" (preemptible)"
    else
        PREEMPTIBLE_NOTE=""
    fi

    TOTAL_COST=$(echo "scale=2; $COMPUTE_COST + $STORAGE_COST" | bc)

    echo ""
    echo "════════════════════════════════════════"
    echo "Vertex AI CPU Cost Estimate"
    echo "════════════════════════════════════════"
    echo ""
    echo "Configuration:"
    echo "  Machine: $MACHINE_NAME"
    echo "  Duration: ${TRAINING_HOURS}h"
    echo ""
    echo "Cost Breakdown:"
    echo "  Compute: \$${COMPUTE_COST}${PREEMPTIBLE_NOTE}"
    echo "  Storage: \$${STORAGE_COST}/month"
    echo ""
    echo "Total Estimated Cost: \$${TOTAL_COST}"
    echo ""
fi

# Cost optimization tips
echo "Cost Optimization Tips:"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
if [ "$PLATFORM" = "bigquery_ml" ]; then
    echo "• Use partitioned tables to reduce data scanned"
    echo "• Filter data in WHERE clause before training"
    echo "• Start with simpler models (LINEAR/LOGISTIC) before AutoML"
    echo "• Use table sampling for quick experiments"
    echo "• Cache intermediate results with materialized views"
else
    echo "• Use preemptible instances (60-70% cost reduction)"
    echo "• Enable checkpointing for preemptible training"
    echo "• Use mixed precision training (faster = cheaper)"
    echo "• Start with smaller GPU/TPU configs for testing"
    echo "• Clean up unused storage and model artifacts"
    echo "• Use Cloud Storage lifecycle policies"
    if [ "$PLATFORM" = "vertex_gpu" ] || [ "$PLATFORM" = "vertex_tpu" ]; then
        echo "• Profile training to eliminate CPU bottlenecks"
        echo "• Consider spot/preemptible pricing (much cheaper)"
    fi
fi
echo ""

# Comparison with other platforms
echo "Platform Comparison:"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
if [ "$PLATFORM" = "bigquery_ml" ]; then
    echo "BigQuery ML vs Alternatives:"
    echo "  BigQuery ML: Best for SQL-based ML, structured data"
    echo "  Vertex AI GPU: Better for deep learning, custom models"
    echo "  Modal/Lambda: Often cheaper for burst GPU workloads"
else
    echo "Vertex AI vs Alternatives:"
    echo "  Vertex AI: Integrated with GCP ecosystem"
    echo "  Modal: \$0.20-4.00/hr GPU, better cold start"
    echo "  Lambda Labs: \$1.10/hr A100, simple pricing"
    echo "  RunPod: \$0.34-2.49/hr, spot pricing available"
fi
echo ""
echo "Note: Prices are approximate and may vary by region."
echo "For exact pricing, see: https://cloud.google.com/products/calculator"
echo ""
