#!/bin/bash
# Setup BigQuery ML environment with authentication and dataset configuration

set -e

echo "=== BigQuery ML Environment Setup ==="
echo ""

# Check if gcloud is installed
if ! command -v gcloud &> /dev/null; then
    echo "Error: gcloud CLI not found. Please install:"
    echo "https://cloud.google.com/sdk/docs/install"
    exit 1
fi

# Check authentication
if ! gcloud auth list --filter=status:ACTIVE --format="value(account)" &> /dev/null; then
    echo "Not authenticated. Running gcloud auth login..."
    gcloud auth login
fi

# Prompt for GCP Project ID
echo "Enter your GCP Project ID:"
read -r PROJECT_ID

if [ -z "$PROJECT_ID" ]; then
    echo "Error: Project ID is required"
    exit 1
fi

# Set project
gcloud config set project "$PROJECT_ID"
echo "✓ Set active project to $PROJECT_ID"

# Enable required APIs
echo ""
echo "Enabling required GCP APIs..."
gcloud services enable bigquery.googleapis.com
gcloud services enable bigquerystorage.googleapis.com
echo "✓ APIs enabled"

# Prompt for BigQuery dataset name
echo ""
echo "Enter BigQuery dataset name (e.g., ml_models):"
read -r DATASET_NAME

if [ -z "$DATASET_NAME" ]; then
    DATASET_NAME="ml_models"
    echo "Using default: $DATASET_NAME"
fi

# Select region
echo ""
echo "Select BigQuery dataset region:"
echo "1) US (multi-region)"
echo "2) EU (multi-region)"
echo "3) us-central1"
echo "4) europe-west4"
echo "5) asia-southeast1"
read -p "Enter choice (1-5): " REGION_CHOICE

case $REGION_CHOICE in
    1) REGION="US" ;;
    2) REGION="EU" ;;
    3) REGION="us-central1" ;;
    4) REGION="europe-west4" ;;
    5) REGION="asia-southeast1" ;;
    *) REGION="US"; echo "Using default: US" ;;
esac

# Create dataset if it doesn't exist
echo ""
echo "Creating BigQuery dataset..."
if bq ls "$DATASET_NAME" &> /dev/null; then
    echo "Dataset $DATASET_NAME already exists"
else
    bq mk --dataset --location="$REGION" "$PROJECT_ID:$DATASET_NAME"
    echo "✓ Created dataset $PROJECT_ID:$DATASET_NAME in $REGION"
fi

# Select default model type preference
echo ""
echo "Select default model type preference:"
echo "1) LINEAR_REG - Linear regression"
echo "2) LOGISTIC_REG - Binary classification"
echo "3) BOOSTED_TREE_CLASSIFIER - XGBoost classification"
echo "4) BOOSTED_TREE_REGRESSOR - XGBoost regression"
echo "5) DNN_CLASSIFIER - Deep neural network classification"
echo "6) DNN_REGRESSOR - Deep neural network regression"
echo "7) AUTOML_CLASSIFIER - AutoML classification"
echo "8) AUTOML_REGRESSOR - AutoML regression"
read -p "Enter choice (1-8): " MODEL_CHOICE

case $MODEL_CHOICE in
    1) DEFAULT_MODEL="LINEAR_REG" ;;
    2) DEFAULT_MODEL="LOGISTIC_REG" ;;
    3) DEFAULT_MODEL="BOOSTED_TREE_CLASSIFIER" ;;
    4) DEFAULT_MODEL="BOOSTED_TREE_REGRESSOR" ;;
    5) DEFAULT_MODEL="DNN_CLASSIFIER" ;;
    6) DEFAULT_MODEL="DNN_REGRESSOR" ;;
    7) DEFAULT_MODEL="AUTOML_CLASSIFIER" ;;
    8) DEFAULT_MODEL="AUTOML_REGRESSOR" ;;
    *) DEFAULT_MODEL="LOGISTIC_REG"; echo "Using default: LOGISTIC_REG" ;;
esac

# Create configuration file
cat > bigquery_config.json << EOF
{
  "project_id": "$PROJECT_ID",
  "dataset_name": "$DATASET_NAME",
  "region": "$REGION",
  "default_model_type": "$DEFAULT_MODEL",
  "billing_project": "$PROJECT_ID"
}
EOF

echo "✓ Created bigquery_config.json"

# Create .bigqueryrc for CLI defaults
cat > .bigqueryrc << EOF
[core]
project_id = $PROJECT_ID

[query]
use_legacy_sql = false

[dataset]
default_dataset = $PROJECT_ID:$DATASET_NAME
EOF

echo "✓ Created .bigqueryrc"

# Verify permissions
echo ""
echo "Verifying IAM permissions..."
ACCOUNT=$(gcloud auth list --filter=status:ACTIVE --format="value(account)")

echo "Checking permissions for $ACCOUNT..."
# Check BigQuery permissions (best effort)
if gcloud projects get-iam-policy "$PROJECT_ID" --flatten="bindings[].members" --filter="bindings.members:$ACCOUNT" --format="value(bindings.role)" | grep -q "bigquery"; then
    echo "✓ BigQuery permissions detected"
else
    echo "⚠ Warning: May need BigQuery permissions. Required roles:"
    echo "  - roles/bigquery.dataEditor"
    echo "  - roles/bigquery.jobUser"
    echo ""
    echo "Grant with:"
    echo "  gcloud projects add-iam-policy-binding $PROJECT_ID \\"
    echo "    --member=user:$ACCOUNT \\"
    echo "    --role=roles/bigquery.dataEditor"
fi

# Create example SQL files directory
mkdir -p examples/bigquery-ml

# Create a simple example query
cat > examples/bigquery-ml/create_model_example.sql << 'EOF'
-- Example: Create a linear regression model
-- Replace with your actual table and columns

CREATE OR REPLACE MODEL `{PROJECT_ID}.{DATASET_NAME}.my_model`
OPTIONS(
  model_type='LINEAR_REG',
  input_label_cols=['label_column'],
  max_iterations=10
) AS
SELECT
  feature1,
  feature2,
  feature3,
  label_column
FROM
  `{PROJECT_ID}.{DATASET_NAME}.your_training_table`
WHERE
  -- Add any filters here
  data_split = 'train';

-- Evaluate the model
SELECT * FROM ML.EVALUATE(
  MODEL `{PROJECT_ID}.{DATASET_NAME}.my_model`,
  (SELECT * FROM `{PROJECT_ID}.{DATASET_NAME}.your_training_table` WHERE data_split = 'test')
);

-- Make predictions
SELECT * FROM ML.PREDICT(
  MODEL `{PROJECT_ID}.{DATASET_NAME}.my_model`,
  (SELECT * FROM `{PROJECT_ID}.{DATASET_NAME}.your_prediction_table`)
);
EOF

sed -i "s/{PROJECT_ID}/$PROJECT_ID/g" examples/bigquery-ml/create_model_example.sql
sed -i "s/{DATASET_NAME}/$DATASET_NAME/g" examples/bigquery-ml/create_model_example.sql

echo "✓ Created example SQL in examples/bigquery-ml/"

# Summary
echo ""
echo "════════════════════════════════════════"
echo "✓ BigQuery ML Setup Complete!"
echo "════════════════════════════════════════"
echo ""
echo "Configuration:"
echo "  Project ID: $PROJECT_ID"
echo "  Dataset: $PROJECT_ID:$DATASET_NAME"
echo "  Region: $REGION"
echo "  Default Model: $DEFAULT_MODEL"
echo ""
echo "Files Created:"
echo "  • bigquery_config.json - Configuration file"
echo "  • .bigqueryrc - CLI defaults"
echo "  • examples/bigquery-ml/create_model_example.sql - Example query"
echo ""
echo "Next Steps:"
echo "  1. Review IAM permissions above"
echo "  2. Prepare your training data in BigQuery"
echo "  3. Copy and modify examples/bigquery-ml/create_model_example.sql"
echo "  4. Run training query in BigQuery console or CLI"
echo "  5. Use ML.EVALUATE() to check model performance"
echo "  6. Use ML.PREDICT() for batch predictions"
echo ""
echo "Useful Commands:"
echo "  bq query --use_legacy_sql=false < your_query.sql"
echo "  bq ls --max_results=10 ml_models"
echo "  bq show --schema ml_models.my_model"
echo ""
echo "Cost Estimation:"
echo "  Run: bash scripts/estimate-gcp-cost.sh"
echo ""
