#!/bin/bash
# Setup Vertex AI environment with GPU/TPU configuration

set -e

echo "=== Vertex AI Environment Setup ==="
echo ""

# Check if gcloud is installed
if ! command -v gcloud &> /dev/null; then
    echo "Error: gcloud CLI not found. Please install:"
    echo "https://cloud.google.com/sdk/docs/install"
    exit 1
fi

# Check authentication
if ! gcloud auth list --filter=status:ACTIVE --format="value(account)" &> /dev/null; then
    echo "Not authenticated. Running gcloud auth login..."
    gcloud auth login
fi

# Prompt for GCP Project ID
echo "Enter your GCP Project ID:"
read -r PROJECT_ID

if [ -z "$PROJECT_ID" ]; then
    echo "Error: Project ID is required"
    exit 1
fi

# Set project
gcloud config set project "$PROJECT_ID"
echo "✓ Set active project to $PROJECT_ID"

# Enable required APIs
echo ""
echo "Enabling required GCP APIs (this may take a few minutes)..."
gcloud services enable aiplatform.googleapis.com
gcloud services enable compute.googleapis.com
gcloud services enable storage-api.googleapis.com
gcloud services enable storage-component.googleapis.com
echo "✓ APIs enabled"

# Select region
echo ""
echo "Select Vertex AI region:"
echo "1) us-central1 (Iowa) - Best GPU availability"
echo "2) us-east1 (South Carolina)"
echo "3) us-west1 (Oregon)"
echo "4) europe-west4 (Netherlands)"
echo "5) asia-southeast1 (Singapore)"
read -p "Enter choice (1-5): " REGION_CHOICE

case $REGION_CHOICE in
    1) REGION="us-central1" ;;
    2) REGION="us-east1" ;;
    3) REGION="us-west1" ;;
    4) REGION="europe-west4" ;;
    5) REGION="asia-southeast1" ;;
    *) REGION="us-central1"; echo "Using default: us-central1" ;;
esac

echo "✓ Selected region: $REGION"

# Create GCS bucket for artifacts
echo ""
echo "Enter GCS bucket name for training artifacts (e.g., ${PROJECT_ID}-vertex-ai):"
read -r BUCKET_NAME

if [ -z "$BUCKET_NAME" ]; then
    BUCKET_NAME="${PROJECT_ID}-vertex-ai"
    echo "Using default: $BUCKET_NAME"
fi

# Check if bucket exists
if gsutil ls "gs://${BUCKET_NAME}" &> /dev/null; then
    echo "Bucket gs://${BUCKET_NAME} already exists"
else
    echo "Creating GCS bucket..."
    gsutil mb -p "$PROJECT_ID" -c STANDARD -l "$REGION" "gs://${BUCKET_NAME}"
    echo "✓ Created bucket gs://${BUCKET_NAME}"
fi

# Select compute type
echo ""
echo "Select default compute type:"
echo "1) CPU - Standard compute (cheapest)"
echo "2) GPU - NVIDIA GPUs for deep learning"
echo "3) TPU - Google TPUs for TensorFlow/JAX"
read -p "Enter choice (1-3): " COMPUTE_CHOICE

case $COMPUTE_CHOICE in
    1) COMPUTE_TYPE="cpu" ;;
    2) COMPUTE_TYPE="gpu" ;;
    3) COMPUTE_TYPE="tpu" ;;
    *) COMPUTE_TYPE="cpu"; echo "Using default: CPU" ;;
esac

# Machine type selection based on compute type
if [ "$COMPUTE_TYPE" = "cpu" ]; then
    echo ""
    echo "Select CPU machine type:"
    echo "1) n1-standard-4 (4 vCPU, 15GB RAM) - $0.19/hr"
    echo "2) n1-standard-8 (8 vCPU, 30GB RAM) - $0.38/hr"
    echo "3) n1-highmem-8 (8 vCPU, 52GB RAM) - $0.47/hr"
    echo "4) n1-highcpu-16 (16 vCPU, 14.4GB RAM) - $0.57/hr"
    read -p "Enter choice (1-4): " MACHINE_CHOICE

    case $MACHINE_CHOICE in
        1) MACHINE_TYPE="n1-standard-4" ;;
        2) MACHINE_TYPE="n1-standard-8" ;;
        3) MACHINE_TYPE="n1-highmem-8" ;;
        4) MACHINE_TYPE="n1-highcpu-16" ;;
        *) MACHINE_TYPE="n1-standard-4" ;;
    esac

    GPU_TYPE="none"
    GPU_COUNT=0
    TPU_TYPE="none"

elif [ "$COMPUTE_TYPE" = "gpu" ]; then
    # Base machine for GPU
    MACHINE_TYPE="n1-standard-8"

    echo ""
    echo "Select GPU type:"
    echo "1) NVIDIA_TESLA_T4 (16GB) - $0.35/hr - Light training/inference"
    echo "2) NVIDIA_TESLA_L4 (24GB) - $0.66/hr - Modern alternative to T4"
    echo "3) NVIDIA_TESLA_V100 (16GB) - $2.48/hr - Mid-size training"
    echo "4) NVIDIA_TESLA_A100 (40GB) - $3.67/hr - Large model training"
    echo "5) NVIDIA_A100_80GB (80GB) - $4.95/hr - Very large models"
    read -p "Enter choice (1-5): " GPU_CHOICE

    case $GPU_CHOICE in
        1) GPU_TYPE="NVIDIA_TESLA_T4" ;;
        2) GPU_TYPE="NVIDIA_TESLA_L4" ;;
        3) GPU_TYPE="NVIDIA_TESLA_V100" ;;
        4) GPU_TYPE="NVIDIA_TESLA_A100" ;;
        5) GPU_TYPE="NVIDIA_A100_80GB" ;;
        *) GPU_TYPE="NVIDIA_TESLA_T4" ;;
    esac

    echo ""
    read -p "Number of GPUs (1-8): " GPU_COUNT
    if [ -z "$GPU_COUNT" ] || [ "$GPU_COUNT" -lt 1 ]; then
        GPU_COUNT=1
    fi
    if [ "$GPU_COUNT" -gt 8 ]; then
        echo "Maximum 8 GPUs supported, setting to 8"
        GPU_COUNT=8
    fi

    TPU_TYPE="none"

else # TPU
    MACHINE_TYPE="none"
    GPU_TYPE="none"
    GPU_COUNT=0

    echo ""
    echo "Select TPU type:"
    echo "1) TPU_V2 (8 cores, 64GB) - $4.50/hr"
    echo "2) TPU_V3 (8 cores, 128GB) - $8.00/hr - Standard choice"
    echo "3) TPU_V4 (8 cores, 256GB) - $11.00/hr - Latest generation"
    echo "4) TPU_V5E (8 cores) - $2.50/hr - Cost-optimized"
    read -p "Enter choice (1-4): " TPU_CHOICE

    case $TPU_CHOICE in
        1) TPU_TYPE="TPU_V2" ;;
        2) TPU_TYPE="TPU_V3" ;;
        3) TPU_TYPE="TPU_V4" ;;
        4) TPU_TYPE="TPU_V5E" ;;
        *) TPU_TYPE="TPU_V3" ;;
    esac
fi

# Python version
echo ""
read -p "Python version for training (default: 3.10): " PYTHON_VERSION
PYTHON_VERSION=${PYTHON_VERSION:-3.10}

# Create Vertex AI configuration file
cat > vertex_config.yaml << EOF
# Vertex AI Training Configuration
project_id: "$PROJECT_ID"
region: "$REGION"
bucket: "gs://${BUCKET_NAME}"

# Compute Configuration
compute_type: "$COMPUTE_TYPE"
machine_type: "$MACHINE_TYPE"

# GPU Configuration
gpu_type: "$GPU_TYPE"
gpu_count: $GPU_COUNT

# TPU Configuration
tpu_type: "$TPU_TYPE"

# Python Configuration
python_version: "$PYTHON_VERSION"

# Training Configuration
checkpoint_dir: "gs://${BUCKET_NAME}/checkpoints"
model_dir: "gs://${BUCKET_NAME}/models"
tensorboard_dir: "gs://${BUCKET_NAME}/tensorboard"
EOF

echo "✓ Created vertex_config.yaml"

# Create requirements.txt template
cat > vertex_requirements.txt << 'EOF'
# Vertex AI Training Requirements
# Add your training dependencies here

# Core ML libraries
torch>=2.0.0
torchvision>=0.15.0
transformers>=4.30.0
datasets>=2.14.0
accelerate>=0.20.0

# Google Cloud
google-cloud-storage>=2.10.0
google-cloud-aiplatform>=1.30.0

# Monitoring
tensorboard>=2.13.0
wandb>=0.15.0

# Utilities
scikit-learn>=1.3.0
pandas>=2.0.0
numpy>=1.24.0
tqdm>=4.65.0
EOF

echo "✓ Created vertex_requirements.txt"

# Check and display quotas
echo ""
echo "Checking GPU/TPU quotas..."
if [ "$COMPUTE_TYPE" = "gpu" ]; then
    echo "Checking GPU quota for $GPU_TYPE..."
    # Note: This is informational, actual quota check requires complex parsing
    echo "To check/request GPU quota:"
    echo "  https://console.cloud.google.com/iam-admin/quotas?project=$PROJECT_ID"
    echo "  Search for: '${GPU_TYPE}' in region '${REGION}'"
elif [ "$COMPUTE_TYPE" = "tpu" ]; then
    echo "Checking TPU quota for $TPU_TYPE..."
    echo "To check/request TPU quota:"
    echo "  https://console.cloud.google.com/iam-admin/quotas?project=$PROJECT_ID"
    echo "  Search for: '${TPU_TYPE}' in region '${REGION}'"
fi

# Verify IAM permissions
echo ""
echo "Verifying IAM permissions..."
ACCOUNT=$(gcloud auth list --filter=status:ACTIVE --format="value(account)")

echo "Checking permissions for $ACCOUNT..."
if gcloud projects get-iam-policy "$PROJECT_ID" --flatten="bindings[].members" --filter="bindings.members:$ACCOUNT" --format="value(bindings.role)" | grep -q "aiplatform"; then
    echo "✓ Vertex AI permissions detected"
else
    echo "⚠ Warning: May need Vertex AI permissions. Required roles:"
    echo "  - roles/aiplatform.user"
    echo "  - roles/storage.objectAdmin"
    echo ""
    echo "Grant with:"
    echo "  gcloud projects add-iam-policy-binding $PROJECT_ID \\"
    echo "    --member=user:$ACCOUNT \\"
    echo "    --role=roles/aiplatform.user"
fi

# Create example training script directory
mkdir -p examples/vertex-ai

# Summary
echo ""
echo "════════════════════════════════════════"
echo "✓ Vertex AI Setup Complete!"
echo "════════════════════════════════════════"
echo ""
echo "Configuration:"
echo "  Project ID: $PROJECT_ID"
echo "  Region: $REGION"
echo "  Bucket: gs://${BUCKET_NAME}"
echo "  Compute: $COMPUTE_TYPE"
if [ "$COMPUTE_TYPE" = "cpu" ]; then
    echo "  Machine Type: $MACHINE_TYPE"
elif [ "$COMPUTE_TYPE" = "gpu" ]; then
    echo "  GPU Type: $GPU_TYPE"
    echo "  GPU Count: $GPU_COUNT"
    echo "  Machine Type: $MACHINE_TYPE"
else
    echo "  TPU Type: $TPU_TYPE"
fi
echo "  Python: $PYTHON_VERSION"
echo ""
echo "Files Created:"
echo "  • vertex_config.yaml - Training configuration"
echo "  • vertex_requirements.txt - Python dependencies"
echo ""
echo "Next Steps:"
echo "  1. Review IAM permissions above"
echo "  2. Check GPU/TPU quota in console (link above)"
echo "  3. Add your training dependencies to vertex_requirements.txt"
echo "  4. Create training script using templates/vertex_training_job.py"
echo "  5. Submit training job with gcloud or Python SDK"
echo ""
echo "Useful Commands:"
echo "  # List training jobs"
echo "  gcloud ai custom-jobs list --region=$REGION"
echo ""
echo "  # Submit training job (after creating training script)"
echo "  gcloud ai custom-jobs create \\"
echo "    --region=$REGION \\"
echo "    --display-name=my-training-job \\"
echo "    --worker-pool-spec=machine-type=$MACHINE_TYPE,replica-count=1,container-image-uri=gcr.io/your-image"
echo ""
echo "Cost Estimation:"
echo "  Run: bash scripts/estimate-gcp-cost.sh"
echo ""
