-- BigQuery ML Training Template
-- Complete SQL template for creating and training ML models in BigQuery

-- ═══════════════════════════════════════════════════════════════════════════
-- CONFIGURATION - Update these variables
-- ═══════════════════════════════════════════════════════════════════════════

DECLARE project_id STRING DEFAULT 'your_project_id_here';
DECLARE dataset_name STRING DEFAULT 'your_dataset_here';
DECLARE model_name STRING DEFAULT 'your_model_name_here';
DECLARE training_table STRING DEFAULT 'your_training_table_here';

-- ═══════════════════════════════════════════════════════════════════════════
-- SECTION 1: LINEAR REGRESSION
-- Use for: Predicting continuous numerical values
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_linear_reg`
OPTIONS(
  model_type='LINEAR_REG',
  input_label_cols=['target_column'],

  -- Optional: Regularization
  l1_reg=0.0,
  l2_reg=0.0,

  -- Optional: Training parameters
  max_iterations=20,
  learn_rate=0.1,
  early_stop=TRUE,
  min_rel_progress=0.01,

  -- Optional: Data split
  data_split_method='AUTO_SPLIT',
  data_split_eval_fraction=0.2,
  data_split_col='data_split'
) AS
SELECT
  feature_1,
  feature_2,
  feature_3,
  -- Add more features here
  target_column
FROM
  `{project_id}.{dataset_name}.{training_table}`
WHERE
  target_column IS NOT NULL;

-- ═══════════════════════════════════════════════════════════════════════════
-- SECTION 2: LOGISTIC REGRESSION (Binary Classification)
-- Use for: Binary classification (yes/no, true/false, 0/1)
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_logistic_reg`
OPTIONS(
  model_type='LOGISTIC_REG',
  input_label_cols=['label_column'],

  -- Optional: Class weights (for imbalanced data)
  auto_class_weights=TRUE,

  -- Optional: Regularization
  l1_reg=0.0,
  l2_reg=0.0,

  -- Optional: Training parameters
  max_iterations=20,
  learn_rate=0.1,
  early_stop=TRUE
) AS
SELECT
  feature_1,
  feature_2,
  feature_3,
  label_column  -- Should be 0/1 or TRUE/FALSE
FROM
  `{project_id}.{dataset_name}.{training_table}`
WHERE
  label_column IS NOT NULL;

-- ═══════════════════════════════════════════════════════════════════════════
-- SECTION 3: BOOSTED TREE CLASSIFIER (XGBoost)
-- Use for: Classification with better accuracy than logistic regression
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_boosted_tree_classifier`
OPTIONS(
  model_type='BOOSTED_TREE_CLASSIFIER',
  input_label_cols=['label_column'],

  -- Optional: Boosting parameters
  booster_type='GBTREE',  -- GBTREE or DART
  num_parallel_tree=1,
  max_iterations=50,
  learning_rate=0.3,
  subsample=0.8,

  -- Optional: Tree parameters
  max_tree_depth=6,
  min_tree_child_weight=1,
  tree_method='AUTO',  -- AUTO, EXACT, APPROX, HIST

  -- Optional: Class weights
  auto_class_weights=TRUE,

  -- Optional: Early stopping
  early_stop=TRUE,
  min_rel_progress=0.01
) AS
SELECT
  feature_1,
  feature_2,
  feature_3,
  label_column
FROM
  `{project_id}.{dataset_name}.{training_table}`
WHERE
  label_column IS NOT NULL;

-- ═══════════════════════════════════════════════════════════════════════════
-- SECTION 4: BOOSTED TREE REGRESSOR
-- Use for: Regression with better accuracy than linear regression
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_boosted_tree_regressor`
OPTIONS(
  model_type='BOOSTED_TREE_REGRESSOR',
  input_label_cols=['target_column'],

  -- Boosting parameters (same as classifier)
  max_iterations=50,
  learning_rate=0.3,
  subsample=0.8,
  max_tree_depth=6
) AS
SELECT
  feature_1,
  feature_2,
  feature_3,
  target_column
FROM
  `{project_id}.{dataset_name}.{training_table}`
WHERE
  target_column IS NOT NULL;

-- ═══════════════════════════════════════════════════════════════════════════
-- SECTION 5: DEEP NEURAL NETWORK (DNN) CLASSIFIER
-- Use for: Complex classification with many features
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_dnn_classifier`
OPTIONS(
  model_type='DNN_CLASSIFIER',
  input_label_cols=['label_column'],

  -- Neural network architecture
  hidden_units=[128, 64, 32],  -- Layer sizes

  -- Training parameters
  max_iterations=50,
  batch_size=32,
  learn_rate=0.001,
  optimizer='ADAM',  -- ADAM, ADAGRAD, SGD, RMSPROP, FTRL

  -- Dropout for regularization
  dropout=0.2,

  -- Activation function
  activation_fn='RELU',  -- RELU, RELU6, CRELU, ELU, SELU

  -- Batch normalization
  enable_global_explain=TRUE
) AS
SELECT
  feature_1,
  feature_2,
  feature_3,
  label_column
FROM
  `{project_id}.{dataset_name}.{training_table}`
WHERE
  label_column IS NOT NULL;

-- ═══════════════════════════════════════════════════════════════════════════
-- SECTION 6: FEATURE ENGINEERING WITH TRANSFORM
-- Advanced: Apply transformations during training
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_with_transform`
TRANSFORM(
  -- Numerical features
  feature_1,
  SQRT(feature_2) AS sqrt_feature_2,
  LOG(feature_3 + 1) AS log_feature_3,

  -- Categorical encoding
  ML.FEATURE_CROSS(STRUCT(category_1, category_2)) AS category_cross,
  ML.BUCKETIZE(numerical_feature, [0, 10, 50, 100]) AS bucketed_feature,

  -- Text features
  ML.NGRAMS(SPLIT(text_column, ' '), [1, 2], ' ') AS text_tokens,

  -- Date/time features
  EXTRACT(DAYOFWEEK FROM date_column) AS day_of_week,
  EXTRACT(HOUR FROM timestamp_column) AS hour_of_day,

  -- Target
  label_column
)
OPTIONS(
  model_type='LOGISTIC_REG',
  input_label_cols=['label_column']
) AS
SELECT
  feature_1,
  feature_2,
  feature_3,
  category_1,
  category_2,
  numerical_feature,
  text_column,
  date_column,
  timestamp_column,
  label_column
FROM
  `{project_id}.{dataset_name}.{training_table}`;

-- ═══════════════════════════════════════════════════════════════════════════
-- SECTION 7: AUTOML TABLES
-- Automatic feature engineering and model selection
-- ═══════════════════════════════════════════════════════════════════════════

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_automl`
OPTIONS(
  model_type='AUTOML_CLASSIFIER',  -- or AUTOML_REGRESSOR
  input_label_cols=['label_column'],

  -- AutoML budget (training time in milliseconds)
  budget_hours=1.0,  -- 1 hour budget

  -- Optimization objective
  optimization_objective='MAXIMIZE_AU_ROC'  -- or MINIMIZE_LOG_LOSS, etc.
) AS
SELECT
  *  -- AutoML automatically selects features
FROM
  `{project_id}.{dataset_name}.{training_table}`;

-- ═══════════════════════════════════════════════════════════════════════════
-- EVALUATION QUERIES
-- ═══════════════════════════════════════════════════════════════════════════

-- Evaluate classifier
SELECT
  *
FROM
  ML.EVALUATE(
    MODEL `{project_id}.{dataset_name}.{model_name}`,
    (SELECT * FROM `{project_id}.{dataset_name}.test_table`)
  );

-- Confusion matrix (classifier only)
SELECT
  *
FROM
  ML.CONFUSION_MATRIX(
    MODEL `{project_id}.{dataset_name}.{model_name}`,
    (SELECT * FROM `{project_id}.{dataset_name}.test_table`)
  );

-- ROC curve (classifier only)
SELECT
  *
FROM
  ML.ROC_CURVE(
    MODEL `{project_id}.{dataset_name}.{model_name}`,
    (SELECT * FROM `{project_id}.{dataset_name}.test_table`)
  );

-- Feature importance
SELECT
  *
FROM
  ML.FEATURE_IMPORTANCE(MODEL `{project_id}.{dataset_name}.{model_name}`);

-- Model weights (for linear/logistic models)
SELECT
  *
FROM
  ML.WEIGHTS(MODEL `{project_id}.{dataset_name}.{model_name}`);

-- Training info
SELECT
  *
FROM
  ML.TRAINING_INFO(MODEL `{project_id}.{dataset_name}.{model_name}`);

-- ═══════════════════════════════════════════════════════════════════════════
-- PREDICTION QUERIES
-- ═══════════════════════════════════════════════════════════════════════════

-- Batch predictions
SELECT
  *
FROM
  ML.PREDICT(
    MODEL `{project_id}.{dataset_name}.{model_name}`,
    (SELECT * FROM `{project_id}.{dataset_name}.prediction_table`)
  );

-- Predictions with probabilities (classifier)
SELECT
  predicted_label,
  predicted_label_probs,
  *
FROM
  ML.PREDICT(
    MODEL `{project_id}.{dataset_name}.{model_name}`,
    (SELECT * FROM `{project_id}.{dataset_name}.prediction_table`)
  );

-- ═══════════════════════════════════════════════════════════════════════════
-- MODEL MANAGEMENT
-- ═══════════════════════════════════════════════════════════════════════════

-- List all models
SELECT
  *
FROM
  `{project_id}.{dataset_name}.INFORMATION_SCHEMA.MODELS`;

-- Drop model
DROP MODEL IF EXISTS `{project_id}.{dataset_name}.{model_name}`;

-- Export model (for use outside BigQuery)
-- Note: Some model types support export to Cloud Storage
EXPORT MODEL `{project_id}.{dataset_name}.{model_name}`
OPTIONS(URI='gs://your_bucket/models/{model_name}');

-- ═══════════════════════════════════════════════════════════════════════════
-- HYPERPARAMETER TUNING
-- Create multiple models with different hyperparameters and compare
-- ═══════════════════════════════════════════════════════════════════════════

-- Example: Test different learning rates
CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_lr_001`
OPTIONS(model_type='LOGISTIC_REG', learn_rate=0.01, input_label_cols=['label_column'])
AS SELECT * FROM `{project_id}.{dataset_name}.{training_table}`;

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_lr_01`
OPTIONS(model_type='LOGISTIC_REG', learn_rate=0.1, input_label_cols=['label_column'])
AS SELECT * FROM `{project_id}.{dataset_name}.{training_table}`;

CREATE OR REPLACE MODEL `{project_id}.{dataset_name}.{model_name}_lr_1`
OPTIONS(model_type='LOGISTIC_REG', learn_rate=1.0, input_label_cols=['label_column'])
AS SELECT * FROM `{project_id}.{dataset_name}.{training_table}`;

-- Compare models
SELECT
  'lr_001' AS model_version,
  *
FROM
  ML.EVALUATE(MODEL `{project_id}.{dataset_name}.{model_name}_lr_001`)
UNION ALL
SELECT
  'lr_01' AS model_version,
  *
FROM
  ML.EVALUATE(MODEL `{project_id}.{dataset_name}.{model_name}_lr_01`)
UNION ALL
SELECT
  'lr_1' AS model_version,
  *
FROM
  ML.EVALUATE(MODEL `{project_id}.{dataset_name}.{model_name}_lr_1`);

-- ═══════════════════════════════════════════════════════════════════════════
-- USAGE NOTES
-- ═══════════════════════════════════════════════════════════════════════════

/*
1. Replace placeholders:
   - {project_id}: Your GCP project ID
   - {dataset_name}: Your BigQuery dataset
   - {model_name}: Your model name
   - {training_table}: Your training data table

2. Choose the right model type:
   - LINEAR_REG: Simple regression, fast training
   - LOGISTIC_REG: Binary classification
   - BOOSTED_TREE: Better accuracy, slower training
   - DNN: Complex patterns, needs more data
   - AUTOML: Automatic optimization, most expensive

3. Feature engineering tips:
   - Handle NULL values (filter or impute)
   - Normalize/standardize numerical features
   - Encode categorical features
   - Create interaction features
   - Extract date/time features

4. Model evaluation:
   - Classification: accuracy, precision, recall, F1, AUC-ROC
   - Regression: MAE, MSE, RMSE, R²
   - Always use hold-out test set

5. Cost optimization:
   - Use partitioned tables
   - Filter data before training
   - Start with simpler models
   - Use table sampling for experiments
*/
