"""
Vertex AI Custom Training Job Template

Complete Python template for custom training on Vertex AI with GPU/TPU support,
distributed training, checkpointing, and metrics logging.

Usage:
    python vertex_training_job.py --config vertex_config.yaml
"""

import os
import argparse
import json
import logging
from pathlib import Path
from typing import Dict, Optional, Tuple

import torch
import torch.nn as nn
import torch.optim as optim
from torch.utils.data import Dataset, DataLoader
from torch.utils.tensorboard import SummaryWriter
from google.cloud import storage
from google.cloud import aiplatform

# Distributed training imports (optional)
try:
    import torch.distributed as dist
    from torch.nn.parallel import DistributedDataParallel as DDP
    DISTRIBUTED_AVAILABLE = True
except ImportError:
    DISTRIBUTED_AVAILABLE = False

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


# ═══════════════════════════════════════════════════════════════════════════
# CONFIGURATION
# ═══════════════════════════════════════════════════════════════════════════

class TrainingConfig:
    """Training configuration"""

    def __init__(self, config_dict: Optional[Dict] = None):
        config = config_dict or {}

        # GCP Configuration
        self.project_id = config.get('project_id', os.getenv('GCP_PROJECT_ID', 'your_project_id_here'))
        self.region = config.get('region', 'us-central1')
        self.bucket = config.get('bucket', f'gs://{self.project_id}-vertex-ai')

        # Model Configuration
        self.model_name = config.get('model_name', 'custom_model')
        self.num_classes = config.get('num_classes', 10)
        self.input_size = config.get('input_size', 784)

        # Training Hyperparameters
        self.batch_size = config.get('batch_size', 32)
        self.num_epochs = config.get('num_epochs', 10)
        self.learning_rate = config.get('learning_rate', 0.001)
        self.weight_decay = config.get('weight_decay', 0.0001)

        # Compute Configuration
        self.num_workers = config.get('num_workers', 4)
        self.use_mixed_precision = config.get('use_mixed_precision', True)

        # Paths
        self.checkpoint_dir = config.get('checkpoint_dir', f'{self.bucket}/checkpoints')
        self.model_dir = config.get('model_dir', f'{self.bucket}/models')
        self.tensorboard_dir = config.get('tensorboard_dir', f'{self.bucket}/tensorboard')

        # Checkpointing
        self.checkpoint_every_n_epochs = config.get('checkpoint_every_n_epochs', 1)
        self.keep_last_n_checkpoints = config.get('keep_last_n_checkpoints', 3)


# ═══════════════════════════════════════════════════════════════════════════
# DATASET (Replace with your own)
# ═══════════════════════════════════════════════════════════════════════════

class CustomDataset(Dataset):
    """
    Custom dataset implementation.
    Replace this with your actual dataset loading logic.
    """

    def __init__(self, data_path: str, split: str = 'train', transform=None):
        """
        Args:
            data_path: Path to dataset (can be GCS path)
            split: 'train', 'val', or 'test'
            transform: Optional transforms
        """
        self.data_path = data_path
        self.split = split
        self.transform = transform

        # Load data from GCS if needed
        if data_path.startswith('gs://'):
            self.data = self._load_from_gcs(data_path)
        else:
            self.data = self._load_from_local(data_path)

    def _load_from_gcs(self, gcs_path: str):
        """Load dataset from Google Cloud Storage"""
        logger.info(f"Loading data from GCS: {gcs_path}")
        # Implement GCS loading logic
        # Example: download files and load
        return []

    def _load_from_local(self, local_path: str):
        """Load dataset from local storage"""
        logger.info(f"Loading data from local: {local_path}")
        # Implement local loading logic
        return []

    def __len__(self):
        return len(self.data)

    def __getitem__(self, idx):
        # Return (input, label) tuple
        sample = self.data[idx]
        if self.transform:
            sample = self.transform(sample)
        return sample


# ═══════════════════════════════════════════════════════════════════════════
# MODEL (Replace with your own)
# ═══════════════════════════════════════════════════════════════════════════

class CustomModel(nn.Module):
    """
    Custom model architecture.
    Replace this with your actual model.
    """

    def __init__(self, input_size: int, num_classes: int):
        super(CustomModel, self).__init__()

        self.fc1 = nn.Linear(input_size, 512)
        self.relu1 = nn.ReLU()
        self.dropout1 = nn.Dropout(0.2)

        self.fc2 = nn.Linear(512, 256)
        self.relu2 = nn.ReLU()
        self.dropout2 = nn.Dropout(0.2)

        self.fc3 = nn.Linear(256, num_classes)

    def forward(self, x):
        x = self.fc1(x)
        x = self.relu1(x)
        x = self.dropout1(x)

        x = self.fc2(x)
        x = self.relu2(x)
        x = self.dropout2(x)

        x = self.fc3(x)
        return x


# ═══════════════════════════════════════════════════════════════════════════
# TRAINING UTILITIES
# ═══════════════════════════════════════════════════════════════════════════

class GCSCheckpointManager:
    """Manage checkpoints in Google Cloud Storage"""

    def __init__(self, checkpoint_dir: str, keep_last_n: int = 3):
        self.checkpoint_dir = checkpoint_dir
        self.keep_last_n = keep_last_n
        self.checkpoint_list = []

        # Parse GCS path
        if checkpoint_dir.startswith('gs://'):
            path_parts = checkpoint_dir.replace('gs://', '').split('/', 1)
            self.bucket_name = path_parts[0]
            self.prefix = path_parts[1] if len(path_parts) > 1 else ''
            self.storage_client = storage.Client()
        else:
            self.bucket_name = None
            self.prefix = checkpoint_dir
            os.makedirs(checkpoint_dir, exist_ok=True)

    def save_checkpoint(self, epoch: int, model: nn.Module, optimizer: optim.Optimizer,
                       metrics: Dict, local_path: str = './checkpoint_temp.pt'):
        """Save checkpoint to GCS"""
        checkpoint = {
            'epoch': epoch,
            'model_state_dict': model.state_dict(),
            'optimizer_state_dict': optimizer.state_dict(),
            'metrics': metrics
        }

        # Save locally first
        torch.save(checkpoint, local_path)

        # Upload to GCS if configured
        if self.bucket_name:
            checkpoint_name = f'checkpoint_epoch_{epoch}.pt'
            blob_path = f'{self.prefix}/{checkpoint_name}' if self.prefix else checkpoint_name

            bucket = self.storage_client.bucket(self.bucket_name)
            blob = bucket.blob(blob_path)
            blob.upload_from_filename(local_path)

            logger.info(f"Checkpoint saved to gs://{self.bucket_name}/{blob_path}")
            self.checkpoint_list.append(blob_path)

            # Clean up old checkpoints
            if len(self.checkpoint_list) > self.keep_last_n:
                old_checkpoint = self.checkpoint_list.pop(0)
                old_blob = bucket.blob(old_checkpoint)
                old_blob.delete()
                logger.info(f"Deleted old checkpoint: {old_checkpoint}")
        else:
            # Local storage
            checkpoint_path = f'{self.prefix}/checkpoint_epoch_{epoch}.pt'
            os.rename(local_path, checkpoint_path)
            logger.info(f"Checkpoint saved to {checkpoint_path}")


def setup_distributed():
    """Setup distributed training"""
    if not DISTRIBUTED_AVAILABLE:
        return 0, 1, False

    # Check if running in distributed mode
    if 'RANK' in os.environ and 'WORLD_SIZE' in os.environ:
        rank = int(os.environ['RANK'])
        world_size = int(os.environ['WORLD_SIZE'])

        # Initialize process group
        dist.init_process_group(backend='nccl')

        # Set device
        torch.cuda.set_device(rank)

        logger.info(f"Distributed training: rank {rank}/{world_size}")
        return rank, world_size, True
    else:
        return 0, 1, False


# ═══════════════════════════════════════════════════════════════════════════
# TRAINING LOOP
# ═══════════════════════════════════════════════════════════════════════════

class Trainer:
    """Main trainer class"""

    def __init__(self, config: TrainingConfig):
        self.config = config

        # Setup distributed training
        self.rank, self.world_size, self.is_distributed = setup_distributed()
        self.is_main_process = (self.rank == 0)

        # Setup device
        if torch.cuda.is_available():
            self.device = torch.device(f'cuda:{self.rank}')
            logger.info(f"Using GPU: {torch.cuda.get_device_name(self.device)}")
        else:
            self.device = torch.device('cpu')
            logger.info("Using CPU")

        # Initialize model
        self.model = CustomModel(
            input_size=config.input_size,
            num_classes=config.num_classes
        ).to(self.device)

        # Wrap model for distributed training
        if self.is_distributed:
            self.model = DDP(self.model, device_ids=[self.rank])

        # Setup optimizer and loss
        self.optimizer = optim.AdamW(
            self.model.parameters(),
            lr=config.learning_rate,
            weight_decay=config.weight_decay
        )
        self.criterion = nn.CrossEntropyLoss()

        # Setup mixed precision training
        self.scaler = torch.cuda.amp.GradScaler() if config.use_mixed_precision else None

        # Setup checkpoint manager
        if self.is_main_process:
            self.checkpoint_manager = GCSCheckpointManager(
                config.checkpoint_dir,
                keep_last_n=config.keep_last_n_checkpoints
            )
            self.writer = SummaryWriter(log_dir='./tensorboard_logs')

    def train_epoch(self, train_loader: DataLoader, epoch: int) -> float:
        """Train for one epoch"""
        self.model.train()
        total_loss = 0.0
        num_batches = len(train_loader)

        for batch_idx, (inputs, targets) in enumerate(train_loader):
            inputs, targets = inputs.to(self.device), targets.to(self.device)

            # Forward pass with mixed precision
            if self.scaler:
                with torch.cuda.amp.autocast():
                    outputs = self.model(inputs)
                    loss = self.criterion(outputs, targets)

                # Backward pass
                self.optimizer.zero_grad()
                self.scaler.scale(loss).backward()
                self.scaler.step(self.optimizer)
                self.scaler.update()
            else:
                outputs = self.model(inputs)
                loss = self.criterion(outputs, targets)

                self.optimizer.zero_grad()
                loss.backward()
                self.optimizer.step()

            total_loss += loss.item()

            # Log progress
            if batch_idx % 100 == 0 and self.is_main_process:
                logger.info(f"Epoch {epoch} [{batch_idx}/{num_batches}] Loss: {loss.item():.4f}")

        avg_loss = total_loss / num_batches
        return avg_loss

    def evaluate(self, val_loader: DataLoader) -> Tuple[float, float]:
        """Evaluate model"""
        self.model.eval()
        total_loss = 0.0
        correct = 0
        total = 0

        with torch.no_grad():
            for inputs, targets in val_loader:
                inputs, targets = inputs.to(self.device), targets.to(self.device)

                outputs = self.model(inputs)
                loss = self.criterion(outputs, targets)

                total_loss += loss.item()
                _, predicted = outputs.max(1)
                total += targets.size(0)
                correct += predicted.eq(targets).sum().item()

        avg_loss = total_loss / len(val_loader)
        accuracy = 100. * correct / total

        return avg_loss, accuracy

    def train(self, train_loader: DataLoader, val_loader: Optional[DataLoader] = None):
        """Main training loop"""
        logger.info("Starting training...")

        for epoch in range(1, self.config.num_epochs + 1):
            logger.info(f"\nEpoch {epoch}/{self.config.num_epochs}")

            # Train
            train_loss = self.train_epoch(train_loader, epoch)

            # Evaluate
            if val_loader and self.is_main_process:
                val_loss, val_accuracy = self.evaluate(val_loader)
                logger.info(f"Train Loss: {train_loss:.4f} | Val Loss: {val_loss:.4f} | Val Acc: {val_accuracy:.2f}%")

                # Log to TensorBoard
                self.writer.add_scalar('Loss/train', train_loss, epoch)
                self.writer.add_scalar('Loss/val', val_loss, epoch)
                self.writer.add_scalar('Accuracy/val', val_accuracy, epoch)
            else:
                logger.info(f"Train Loss: {train_loss:.4f}")

            # Save checkpoint
            if self.is_main_process and epoch % self.config.checkpoint_every_n_epochs == 0:
                metrics = {
                    'train_loss': train_loss,
                    'val_loss': val_loss if val_loader else None,
                    'val_accuracy': val_accuracy if val_loader else None
                }
                self.checkpoint_manager.save_checkpoint(epoch, self.model, self.optimizer, metrics)

        logger.info("Training complete!")

        if self.is_main_process:
            self.writer.close()


# ═══════════════════════════════════════════════════════════════════════════
# MAIN
# ═══════════════════════════════════════════════════════════════════════════

def main():
    parser = argparse.ArgumentParser(description='Vertex AI Custom Training')
    parser.add_argument('--config', type=str, default='vertex_config.yaml', help='Path to config file')
    parser.add_argument('--data_path', type=str, default='gs://your_bucket/data', help='Path to training data')
    args = parser.parse_args()

    # Load configuration
    # For YAML config: import yaml; config = yaml.safe_load(open(args.config))
    config = TrainingConfig()

    # Create datasets
    train_dataset = CustomDataset(args.data_path, split='train')
    val_dataset = CustomDataset(args.data_path, split='val')

    # Create data loaders
    train_loader = DataLoader(
        train_dataset,
        batch_size=config.batch_size,
        shuffle=True,
        num_workers=config.num_workers,
        pin_memory=True
    )

    val_loader = DataLoader(
        val_dataset,
        batch_size=config.batch_size,
        shuffle=False,
        num_workers=config.num_workers,
        pin_memory=True
    )

    # Create trainer and train
    trainer = Trainer(config)
    trainer.train(train_loader, val_loader)


if __name__ == '__main__':
    main()
