# Data Validation Example

This example demonstrates how to validate training datasets before fine-tuning a language model.

## Scenario

You're preparing to fine-tune a Llama 2 7B model for sentiment analysis using a custom dataset of product reviews. Before investing time and money in training, you want to ensure:

1. Data format is correct (JSONL)
2. All required fields are present
3. Text lengths are within model limits
4. No duplicate entries exist
5. Label distribution is reasonable
6. Token counts are within context window

## Dataset Overview

- **Format**: JSONL (JSON Lines)
- **Size**: 10,000 training samples
- **Task**: Sentiment classification (positive/negative/neutral)
- **Fields**: `text`, `label`, `metadata`

## Sample Data

```jsonl
{"text": "This product exceeded my expectations! The quality is outstanding.", "label": "positive", "metadata": {"source": "amazon", "timestamp": "2025-01-15T10:30:00Z"}}
{"text": "Terrible experience. The item broke after two days.", "label": "negative", "metadata": {"source": "amazon", "timestamp": "2025-01-15T11:45:00Z"}}
{"text": "Product arrived on time and matches the description.", "label": "neutral", "metadata": {"source": "amazon", "timestamp": "2025-01-15T14:20:00Z"}}
```

## Step 1: Create Validation Schema

Create a JSON schema that defines your data requirements:

```bash
cat > validation-schema.json << 'EOF'
{
  "$schema": "http://json-schema.org/draft-07/schema#",
  "type": "object",
  "required": ["text", "label"],
  "properties": {
    "text": {
      "type": "string",
      "minLength": 10,
      "maxLength": 5000
    },
    "label": {
      "type": "string",
      "enum": ["positive", "negative", "neutral"]
    },
    "metadata": {
      "type": "object",
      "properties": {
        "source": {"type": "string"},
        "timestamp": {"type": "string", "format": "date-time"}
      }
    }
  },
  "validation_rules": {
    "max_token_length": 2048,
    "tokenizer": "meta-llama/Llama-2-7b-hf",
    "check_duplicates": true,
    "min_label_count": 100,
    "max_label_imbalance_ratio": 5.0
  }
}
EOF
```

## Step 2: Run Basic Validation

Validate the data format and schema compliance:

```bash
bash scripts/validate-data.sh \
  --data-path ./data/train.jsonl \
  --format jsonl \
  --schema validation-schema.json \
  --output validation-report.json
```

**Output:**
```
[INFO] Starting validation with format: jsonl
[INFO] Validating JSONL format...

Validation Summary:
Total samples: 10000
Valid samples: 9987
Invalid samples: 13
Errors found: 13

[INFO] Validation report saved to: validation-report.json
[ERROR] ✗ Validation FAILED
```

## Step 3: Review Validation Errors

Check the validation report to see what went wrong:

```bash
cat validation-report.json | jq '.validation_errors[:5]'
```

**Output:**
```json
[
  {
    "line": 42,
    "field": "text",
    "error": "Exceeds max token length: 2150 > 2048"
  },
  {
    "line": 156,
    "field": "label",
    "error": "Required field 'label' missing"
  },
  {
    "line": 387,
    "field": "label",
    "error": "Invalid label value: 'unknwn' (typo)"
  },
  {
    "line": 892,
    "field": "text",
    "error": "Text length too short: 8 < 10"
  },
  {
    "line": 1423,
    "error": "Invalid JSON: Expecting ',' delimiter"
  }
]
```

## Step 4: Run Comprehensive Validation

Add duplicate checking and token validation:

```bash
bash scripts/validate-data.sh \
  --data-path ./data/train.jsonl \
  --schema validation-schema.json \
  --check-duplicates \
  --check-null \
  --check-tokens \
  --tokenizer meta-llama/Llama-2-7b-hf \
  --max-length 2048 \
  --output full-validation-report.json \
  --verbose
```

**Output:**
```
[INFO] Starting validation with format: jsonl
[INFO] Validating JSONL format...

Validation Summary:
Total samples: 10000
Valid samples: 9850
Invalid samples: 150
Errors found: 178
Duplicates found: 28

[INFO] Validation report saved to: full-validation-report.json
[ERROR] ✗ Validation FAILED
```

## Step 5: Analyze Statistics

Check the data statistics and recommendations:

```bash
cat full-validation-report.json | jq '.statistics'
```

**Output:**
```json
{
  "avg_text_length": 487,
  "min_text_length": 8,
  "max_text_length": 8432,
  "avg_token_count": 128,
  "min_token_count": 2,
  "max_token_count": 2150,
  "label_distribution": {
    "positive": 4892,
    "negative": 4895,
    "neutral": 213
  }
}
```

**Analysis:**
- **Good**: Average text length (487 chars) and token count (128 tokens) are reasonable
- **Good**: Positive/negative labels are well balanced (4892 vs 4895)
- **Problem**: Neutral class is severely underrepresented (213 vs ~4900)
- **Problem**: Some texts exceed max token length (2150 > 2048)
- **Problem**: Some texts are too short (8 < 10)

## Step 6: Review Recommendations

```bash
cat full-validation-report.json | jq '.recommendations'
```

**Output:**
```json
[
  "Fix 178 validation errors before training",
  "Found 28 duplicate entries - consider deduplication",
  "Label distribution is imbalanced - consider class weighting",
  "28 samples exceed max token length - truncate or filter",
  "15 samples below minimum length - remove or expand"
]
```

## Step 7: Fix Data Issues

Based on the validation report, fix the issues:

### Fix 1: Remove duplicates
```bash
# Python script to deduplicate
python3 - << 'PYTHON'
import json
import hashlib

seen = set()
with open('./data/train.jsonl', 'r') as f_in, \
     open('./data/train_deduped.jsonl', 'w') as f_out:
    for line in f_in:
        obj = json.loads(line)
        content_hash = hashlib.md5(
            json.dumps(obj, sort_keys=True).encode()
        ).hexdigest()
        if content_hash not in seen:
            seen.add(content_hash)
            f_out.write(line)
PYTHON
```

### Fix 2: Filter out invalid samples
```bash
# Python script to filter
python3 - << 'PYTHON'
import json

with open('./data/train_deduped.jsonl', 'r') as f_in, \
     open('./data/train_clean.jsonl', 'w') as f_out:
    for line in f_in:
        try:
            obj = json.loads(line)
            # Check required fields
            if 'text' not in obj or 'label' not in obj:
                continue
            # Check text length
            if len(obj['text']) < 10 or len(obj['text']) > 5000:
                continue
            # Check valid label
            if obj['label'] not in ['positive', 'negative', 'neutral']:
                continue
            f_out.write(line)
        except:
            continue
PYTHON
```

### Fix 3: Address class imbalance
```bash
# Add class weights to training config
cat >> training_config.yaml << 'EOF'
class_weights:
  positive: 1.0
  negative: 1.0
  neutral: 20.0  # Upweight underrepresented class
EOF
```

## Step 8: Re-validate Clean Data

```bash
bash scripts/validate-data.sh \
  --data-path ./data/train_clean.jsonl \
  --schema validation-schema.json \
  --check-duplicates \
  --check-tokens \
  --max-length 2048 \
  --output final-validation-report.json
```

**Output:**
```
[INFO] Starting validation with format: jsonl
[INFO] Validating JSONL format...

Validation Summary:
Total samples: 9972
Valid samples: 9972
Invalid samples: 0
Errors found: 0
Duplicates found: 0

[INFO] Validation report saved to: final-validation-report.json
[INFO] ✓ Validation PASSED
```

## Step 9: Final Statistics

```bash
cat final-validation-report.json | jq '.statistics'
```

**Output:**
```json
{
  "avg_text_length": 485,
  "min_text_length": 10,
  "max_text_length": 2048,
  "avg_token_count": 127,
  "min_token_count": 3,
  "max_token_count": 512,
  "label_distribution": {
    "positive": 4892,
    "negative": 4895,
    "neutral": 185
  }
}
```

**✓ Ready for Training:**
- All samples are valid
- No duplicates
- All token counts within limits
- Class imbalance addressed via weighting

## Results Summary

| Metric | Before Validation | After Cleanup |
|--------|------------------|---------------|
| Total Samples | 10,000 | 9,972 |
| Valid Samples | 9,850 | 9,972 |
| Duplicates | 28 | 0 |
| Validation Errors | 178 | 0 |
| Max Token Length | 2,150 | 512 |

**Time Saved**: By catching these issues before training, you avoided:
- Wasted GPU hours on invalid data
- Training failures due to token length issues
- Poor model performance due to duplicates
- Unbalanced predictions due to class imbalance

**Next Steps:**
1. Proceed with training on `train_clean.jsonl`
2. Use class weights to handle imbalance
3. Monitor training metrics closely
4. Validate test/validation sets using the same process

## Best Practices Learned

1. **Always validate before training** - Catches issues early
2. **Use comprehensive checks** - Format, schema, duplicates, tokens
3. **Review statistics** - Understand your data distribution
4. **Fix systematically** - Address one issue at a time
5. **Re-validate after fixes** - Ensure cleanup worked correctly
6. **Document the process** - Keep validation reports for reproducibility
