# Pipeline Testing Example

This example demonstrates how to test an end-to-end ML training pipeline before running full-scale training.

## Scenario

You've built a complete fine-tuning pipeline for Llama 2 7B using:
- **Platform**: Modal (serverless GPU)
- **Framework**: PyTorch + HuggingFace Transformers
- **Method**: LoRA/PEFT fine-tuning
- **Dataset**: 10,000 sentiment analysis samples
- **GPU**: T4 (16GB VRAM)

Before launching a full training job that will cost $10-20, you want to verify the entire pipeline works correctly with a small sample.

## Pipeline Components

```
Data Loading → Tokenization → Model Loading → Training Loop →
Validation → Checkpoint Saving → Checkpoint Loading → Inference
```

## Step 1: Prepare Test Configuration

Create a test configuration that uses small samples and few steps:

```bash
cat > test-config.yaml << 'EOF'
pipeline:
  name: llama-7b-sentiment-test
  framework: pytorch
  platform: modal

data:
  train_path: ./data/sample-train.jsonl
  val_path: ./data/sample-val.jsonl
  format: jsonl
  sample_size: 10  # Only 10 samples for testing

model:
  base_model: meta-llama/Llama-2-7b-hf
  checkpoint_path: ./checkpoints/test
  lora:
    enabled: true
    r: 16
    alpha: 32
    dropout: 0.05
    target_modules: ["q_proj", "v_proj"]

training:
  batch_size: 1
  gradient_accumulation: 1
  learning_rate: 2e-4
  max_steps: 5  # Only 5 steps for testing
  warmup_steps: 0
  eval_steps: 2

testing:
  sample_size: 10
  timeout_seconds: 300
  expected_loss_range: [0.5, 3.0]
  fail_fast: true
  cleanup: true

gpu:
  required: true
  min_vram_gb: 16
  allow_cpu_fallback: false
EOF
```

## Step 2: Create Sample Data

Extract a small sample from your training data for testing:

```bash
# Take first 10 lines for training
head -10 ./data/train.jsonl > ./data/sample-train.jsonl

# Take first 5 lines for validation
head -5 ./data/val.jsonl > ./data/sample-val.jsonl
```

**Sample data:**
```jsonl
{"text": "This product is amazing! Highly recommend it.", "label": "positive"}
{"text": "Terrible quality. Complete waste of money.", "label": "negative"}
{"text": "It's okay, nothing special.", "label": "neutral"}
```

## Step 3: Check Dependencies

First, verify all required dependencies are available:

```bash
bash scripts/check-dependencies.sh \
  --framework pytorch \
  --gpu-required \
  --min-vram 16 \
  --packages requirements.txt \
  --output dependency-check.json
```

**Output:**
```
[INFO] Checking dependencies for platform: local

Dependency Check Summary:
=========================
python: PASS
framework: PASS
gpu: PASS
packages: PASS
storage: PASS

[INFO] ✓ All dependencies satisfied
```

**Dependency report:**
```json
{
  "status": "PASS",
  "platform": "local",
  "checks": {
    "python": {
      "status": "PASS",
      "version": "3.10.12",
      "required": ">=3.8"
    },
    "framework": {
      "status": "PASS",
      "version": "2.1.0",
      "cuda_available": true,
      "cuda_version": "12.1"
    },
    "gpu": {
      "status": "PASS",
      "count": 1,
      "gpus": [
        {
          "name": "NVIDIA A100-SXM4-40GB",
          "vram_gb": 40,
          "driver_version": "535.129.03"
        }
      ]
    },
    "packages": {
      "status": "PASS",
      "installed": 42,
      "missing": 0,
      "outdated": 3
    },
    "storage": {
      "status": "PASS",
      "available_gb": 128,
      "required_gb": 50
    }
  }
}
```

**✓ Dependencies OK - Proceed to pipeline testing**

## Step 4: Run Complete Pipeline Test

Test all pipeline steps with sample data:

```bash
bash scripts/test-pipeline.sh \
  --config test-config.yaml \
  --verbose \
  --fail-fast \
  --output pipeline-test-report.txt
```

**Output:**
```
[INFO] Loading configuration: test-config.yaml
[INFO] Starting pipeline tests...

Pipeline Test Report
====================

Pipeline: llama-7b-sentiment-test
Started: 2025-11-01 14:30:45

Test Results:
✓ data_loading (2.3s) - Loaded 10 samples successfully
✓ tokenization (1.1s) - Tokenized all samples, avg length: 128 tokens
✓ model_loading (8.7s) - Model loaded, 7.2B parameters
✓ training_step (15.4s) - Training step completed, loss: 1.847
✓ validation_step (12.1s) - Validation step completed, loss: 1.923
✓ checkpoint_save (3.2s) - Checkpoint saved to ./checkpoints/test
✓ checkpoint_load (6.8s) - Checkpoint loaded successfully
✓ inference (2.9s) - Inference completed, latency: 142ms
✓ metrics (0.4s) - Metrics calculated correctly

Overall: PASS (9/9 tests passed)

Performance Metrics:
- Total time: 52s
- GPU memory used: 15.2 GB
- CPU memory used: 8.4 GB

Recommendations:
- Pipeline is ready for full training
- Consider increasing batch_size to improve throughput
```

**✓ All tests passed!** The pipeline is working correctly.

## Step 5: Test Individual Components

If any step fails, you can test components individually:

### Test Data Loading Only
```bash
bash scripts/test-pipeline.sh \
  --config test-config.yaml \
  --steps data_loading,tokenization
```

### Test Training Steps Only
```bash
bash scripts/test-pipeline.sh \
  --config test-config.yaml \
  --steps training_step,validation_step
```

### Test Inference Pipeline
```bash
bash scripts/test-pipeline.sh \
  --config test-config.yaml \
  --steps model_loading,inference
```

## Step 6: Validate Model Checkpoint

After the pipeline test creates a checkpoint, validate it:

```bash
bash scripts/validate-model.sh \
  --model-path ./checkpoints/test \
  --framework pytorch \
  --check-weights \
  --check-config \
  --check-tokenizer \
  --check-inference \
  --sample-input "This product is great!" \
  --output model-validation-report.json
```

**Output:**
```
[INFO] Checking model directory structure...
[INFO] Validation report saved to: model-validation-report.json
Checking file structure...
Checking model configuration...
Checking model weights...
Checking tokenizer files...
Running inference test...

Validation Summary:
Status: PASS
  file_structure: PASS
  config: PASS
  weights: PASS
  tokenizer: PASS
  inference: PASS

[INFO] ✓ Model validation PASSED
```

**Model validation report:**
```json
{
  "status": "PASS",
  "model_path": "./checkpoints/test",
  "framework": "pytorch",
  "checks": {
    "file_structure": "PASS",
    "config": "PASS",
    "weights": "PASS",
    "tokenizer": "PASS",
    "inference": "PASS"
  },
  "model_info": {
    "architecture": "LlamaForCausalLM",
    "parameters": "7.2B",
    "lora_enabled": true,
    "lora_rank": 16,
    "vocab_size": 32000
  },
  "memory_estimate": {
    "model_size_gb": 13.5,
    "inference_vram_gb": 16.2,
    "training_vram_gb": 24.8
  },
  "inference_test": {
    "input": "This product is great!",
    "output": "This product is great! I love the quality and features.",
    "latency_ms": 142
  }
}
```

## Step 7: Performance Analysis

Based on the test results, estimate full training performance:

**Test Performance:**
- 5 training steps on 10 samples: 15.4s
- Steps per second: 0.32
- GPU memory used: 15.2 GB

**Full Training Projection:**
- Total samples: 10,000
- Epochs: 3
- Batch size: 1
- Total steps: 30,000

**Time Estimate:**
- At 0.32 steps/sec: ~26 hours
- **Issue detected**: Too slow!

**Optimization Applied:**
```yaml
training:
  batch_size: 4  # Increase from 1
  gradient_accumulation: 4  # Add gradient accumulation
  # Effective batch size: 16
```

## Step 8: Re-test with Optimizations

```bash
# Update test config with optimizations
sed -i 's/batch_size: 1/batch_size: 4/' test-config.yaml
sed -i 's/gradient_accumulation: 1/gradient_accumulation: 4/' test-config.yaml

# Re-run pipeline test
bash scripts/test-pipeline.sh \
  --config test-config.yaml \
  --steps training_step \
  --verbose
```

**Output:**
```
✓ training_step (8.2s) - Training step completed, loss: 1.834

Performance Metrics:
- Steps per second: 0.61 (was 0.32)
- GPU memory used: 18.9 GB (was 15.2 GB)
- Speedup: 1.9x
```

**New projection:**
- At 0.61 steps/sec: ~13.7 hours
- **Much better!** Cut training time in half.

## Step 9: Troubleshooting Example

### Scenario: Pipeline Test Fails

**Error encountered:**
```
✗ training_step - Error: CUDA out of memory. Tried to allocate 2.00 GiB
```

**Fix applied:**
```yaml
training:
  batch_size: 2  # Reduce from 4
  gradient_accumulation: 8  # Increase to maintain effective batch size

model:
  load_in_8bit: true  # Enable 8-bit quantization
  lora:
    enabled: true
    r: 8  # Reduce from 16 to save memory
```

**Re-test:**
```bash
bash scripts/test-pipeline.sh \
  --config test-config.yaml \
  --steps training_step
```

**Output:**
```
✓ training_step (10.1s) - Training step completed, loss: 1.862

Performance Metrics:
- GPU memory used: 14.3 GB (within 16GB limit)
```

**✓ Fixed!** Training now fits in GPU memory.

## Step 10: Final Validation Before Full Training

Run a comprehensive pre-flight check:

```bash
# 1. Validate training data
bash scripts/validate-data.sh \
  --data-path ./data/train.jsonl \
  --check-duplicates \
  --check-tokens \
  --max-length 2048

# 2. Check dependencies
bash scripts/check-dependencies.sh \
  --framework pytorch \
  --gpu-required \
  --min-vram 16

# 3. Test complete pipeline
bash scripts/test-pipeline.sh \
  --config test-config.yaml \
  --verbose
```

**All checks passed!** ✓

## Results Summary

| Step | Status | Time | Notes |
|------|--------|------|-------|
| Data Loading | ✓ PASS | 2.3s | 10 samples loaded |
| Tokenization | ✓ PASS | 1.1s | Avg 128 tokens |
| Model Loading | ✓ PASS | 8.7s | 7.2B params |
| Training Step | ✓ PASS | 8.2s | Loss: 1.834 |
| Validation Step | ✓ PASS | 12.1s | Loss: 1.923 |
| Checkpoint Save | ✓ PASS | 3.2s | Saved to disk |
| Checkpoint Load | ✓ PASS | 6.8s | Loaded successfully |
| Inference | ✓ PASS | 2.9s | 142ms latency |
| Metrics | ✓ PASS | 0.4s | Calculated correctly |

**Total Pipeline Test Time**: 52 seconds
**Issues Found**: 1 (memory overflow)
**Issues Fixed**: 1 (reduced batch size, enabled quantization)
**Ready for Production**: ✓ Yes

## Cost Savings

**Without pipeline testing:**
- Launch full training blindly
- Hit memory error after 2 hours
- Debug and restart
- Total wasted: 2 GPU hours = $1.18

**With pipeline testing:**
- Caught memory issue in 52 seconds
- Fixed before full training
- Saved: $1.18 and 2 hours of time

**ROI**: Pipeline testing paid for itself immediately!

## Key Takeaways

1. **Always test before full training** - Catches configuration errors early
2. **Use small samples** - Test with 10-20 samples for fast iteration
3. **Test all components** - Data, model, training, inference
4. **Check performance** - Estimate full training time from test results
5. **Iterate on config** - Optimize batch size, memory usage based on tests
6. **Validate checkpoints** - Ensure model saves/loads correctly
7. **Document issues** - Keep track of problems and solutions

## Next Steps

Now that the pipeline is validated:

1. ✓ Data is validated and clean
2. ✓ Dependencies are satisfied
3. ✓ Pipeline components work correctly
4. ✓ Performance is optimized
5. ✓ Model checkpoints are valid

**Ready to launch full training!**

```bash
# Launch full training with confidence
modal run train.py --config production-config.yaml
```
