#!/usr/bin/env bash

# check-dependencies.sh - System dependency and package verification
# Validates all required dependencies for ML training

set -euo pipefail

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# Default values
FRAMEWORK="pytorch"
GPU_REQUIRED=false
MIN_VRAM=0
CUDA_VERSION=""
PACKAGES=""
PLATFORM="local"
FIX=false
OUTPUT="dependency-report.json"

# Usage
usage() {
    cat << EOF
Usage: check-dependencies.sh [OPTIONS]

Verify all required system dependencies, packages, and GPU availability.

Options:
    --framework FRAMEWORK   ML framework: pytorch, tensorflow, jax (default: pytorch)
    --gpu-required          Require GPU availability
    --min-vram GB           Minimum GPU VRAM required (GB)
    --cuda-version VERSION  Required CUDA version
    --packages FILE         Path to requirements.txt or packages list
    --platform PLATFORM     Platform: modal, lambda, runpod, local (default: local)
    --fix                   Attempt to install missing packages
    --output REPORT         Output dependency report path
    -h, --help              Show this help message

Examples:
    # Basic check
    bash check-dependencies.sh --framework pytorch

    # Full check with GPU requirements
    bash check-dependencies.sh \\
        --framework pytorch \\
        --gpu-required \\
        --min-vram 16 \\
        --packages requirements.txt

    # Check and fix missing dependencies
    bash check-dependencies.sh --framework pytorch --fix

Exit Codes:
    0 - All dependencies met
    1 - Missing dependencies
    2 - Script error
EOF
}

# Logging
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1" >&2
}

# Check Python version
check_python() {
    if ! command -v python3 &> /dev/null; then
        echo '{"status": "FAIL", "version": "not found", "required": ">=3.8"}'
        return 1
    fi

    local version=$(python3 --version 2>&1 | awk '{print $2}')
    local major=$(echo "$version" | cut -d. -f1)
    local minor=$(echo "$version" | cut -d. -f2)

    if [[ $major -lt 3 ]] || { [[ $major -eq 3 ]] && [[ $minor -lt 8 ]]; }; then
        echo "{\"status\": \"FAIL\", \"version\": \"$version\", \"required\": \">=3.8\"}"
        return 1
    fi

    echo "{\"status\": \"PASS\", \"version\": \"$version\", \"required\": \">=3.8\"}"
    return 0
}

# Check ML framework
check_framework() {
    local framework="$1"

    python3 - "$framework" << 'PYTHON_SCRIPT'
import sys
import json

framework = sys.argv[1]
result = {"status": "FAIL"}

try:
    if framework == "pytorch":
        import torch
        result["status"] = "PASS"
        result["version"] = torch.__version__
        result["cuda_available"] = torch.cuda.is_available()
        if torch.cuda.is_available():
            result["cuda_version"] = torch.version.cuda
    elif framework == "tensorflow":
        import tensorflow as tf
        result["status"] = "PASS"
        result["version"] = tf.__version__
        result["cuda_available"] = len(tf.config.list_physical_devices('GPU')) > 0
    elif framework == "jax":
        import jax
        result["status"] = "PASS"
        result["version"] = jax.__version__
        result["cuda_available"] = any('gpu' in str(d).lower() for d in jax.devices())
    else:
        result["error"] = f"Unknown framework: {framework}"
except ImportError as e:
    result["error"] = str(e)

print(json.dumps(result))
PYTHON_SCRIPT
}

# Check GPU availability
check_gpu() {
    python3 << 'PYTHON_SCRIPT'
import json
import subprocess

result = {
    "status": "FAIL",
    "count": 0,
    "gpus": []
}

try:
    # Try nvidia-smi
    output = subprocess.check_output(
        ["nvidia-smi", "--query-gpu=name,memory.total,driver_version", "--format=csv,noheader"],
        stderr=subprocess.DEVNULL
    ).decode()

    gpus = []
    for line in output.strip().split('\n'):
        parts = [p.strip() for p in line.split(',')]
        if len(parts) >= 2:
            name = parts[0]
            memory_str = parts[1]
            # Extract memory in GB
            vram_gb = int(memory_str.split()[0]) // 1024 if 'MiB' in memory_str else int(memory_str.split()[0])
            driver = parts[2] if len(parts) > 2 else "unknown"

            gpus.append({
                "name": name,
                "vram_gb": vram_gb,
                "driver_version": driver
            })

    if gpus:
        result["status"] = "PASS"
        result["count"] = len(gpus)
        result["gpus"] = gpus

except (FileNotFoundError, subprocess.CalledProcessError):
    result["error"] = "nvidia-smi not found or failed"

print(json.dumps(result))
PYTHON_SCRIPT
}

# Check Python packages
check_packages() {
    local packages_file="$1"

    if [[ ! -f "$packages_file" ]]; then
        echo '{"status": "SKIP", "reason": "No packages file provided"}'
        return 0
    fi

    python3 - "$packages_file" << 'PYTHON_SCRIPT'
import sys
import json
import pkg_resources
from pathlib import Path

packages_file = sys.argv[1]
result = {
    "status": "PASS",
    "installed": 0,
    "missing": 0,
    "outdated": 0,
    "missing_packages": [],
    "outdated_packages": []
}

# Read requirements
with open(packages_file, 'r') as f:
    requirements = [line.strip() for line in f if line.strip() and not line.startswith('#')]

for req in requirements:
    # Simple parsing - just package name
    pkg_name = req.split('==')[0].split('>=')[0].split('<=')[0].strip()

    try:
        pkg_resources.require(pkg_name)
        result["installed"] += 1
    except pkg_resources.DistributionNotFound:
        result["missing"] += 1
        result["missing_packages"].append(pkg_name)
        result["status"] = "FAIL"
    except pkg_resources.VersionConflict:
        result["outdated"] += 1
        result["outdated_packages"].append(pkg_name)

print(json.dumps(result))
PYTHON_SCRIPT
}

# Check storage
check_storage() {
    local available_gb=$(df -BG . | tail -1 | awk '{print $4}' | sed 's/G//')
    local required_gb=50

    if [[ $available_gb -lt $required_gb ]]; then
        echo "{\"status\": \"FAIL\", \"available_gb\": $available_gb, \"required_gb\": $required_gb}"
        return 1
    fi

    echo "{\"status\": \"PASS\", \"available_gb\": $available_gb, \"required_gb\": $required_gb}"
    return 0
}

# Generate full report
generate_report() {
    local python_check=$(check_python)
    local framework_check=$(check_framework "$FRAMEWORK")
    local gpu_check=$(check_gpu)
    local storage_check=$(check_storage)

    local packages_check='{}'
    if [[ -n "$PACKAGES" ]]; then
        packages_check=$(check_packages "$PACKAGES")
    fi

    # Build JSON report
    cat > "$OUTPUT" << EOF
{
  "platform": "$PLATFORM",
  "timestamp": "$(date -u +%Y-%m-%dT%H:%M:%SZ)",
  "checks": {
    "python": $python_check,
    "framework": $framework_check,
    "gpu": $gpu_check,
    "storage": $storage_check
EOF

    if [[ -n "$PACKAGES" ]]; then
        echo "    ,\"packages\": $packages_check" >> "$OUTPUT"
    fi

    cat >> "$OUTPUT" << 'EOF'
  },
  "recommendations": []
}
EOF

    # Determine overall status
    local python_status=$(echo "$python_check" | jq -r '.status')
    local framework_status=$(echo "$framework_check" | jq -r '.status')
    local gpu_status=$(echo "$gpu_check" | jq -r '.status')
    local storage_status=$(echo "$storage_check" | jq -r '.status')

    local overall_status="PASS"

    if [[ "$python_status" == "FAIL" ]]; then
        overall_status="FAIL"
    fi

    if [[ "$framework_status" == "FAIL" ]]; then
        overall_status="FAIL"
    fi

    if [[ "$GPU_REQUIRED" == true ]] && [[ "$gpu_status" == "FAIL" ]]; then
        overall_status="FAIL"
    fi

    if [[ "$storage_status" == "FAIL" ]]; then
        overall_status="FAIL"
    fi

    # Add overall status
    local temp_file=$(mktemp)
    jq --arg status "$overall_status" '. + {status: $status}' "$OUTPUT" > "$temp_file"
    mv "$temp_file" "$OUTPUT"
}

# Main
main() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --framework)
                FRAMEWORK="$2"
                shift 2
                ;;
            --gpu-required)
                GPU_REQUIRED=true
                shift
                ;;
            --min-vram)
                MIN_VRAM="$2"
                shift 2
                ;;
            --cuda-version)
                CUDA_VERSION="$2"
                shift 2
                ;;
            --packages)
                PACKAGES="$2"
                shift 2
                ;;
            --platform)
                PLATFORM="$2"
                shift 2
                ;;
            --fix)
                FIX=true
                shift
                ;;
            --output)
                OUTPUT="$2"
                shift 2
                ;;
            -h|--help)
                usage
                exit 0
                ;;
            *)
                log_error "Unknown option: $1"
                usage
                exit 2
                ;;
        esac
    done

    log_info "Checking dependencies for platform: $PLATFORM"

    # Generate report
    generate_report

    # Display results
    if command -v jq &> /dev/null && [[ -f "$OUTPUT" ]]; then
        log_info "Dependency report saved to: $OUTPUT"

        local status=$(jq -r '.status' "$OUTPUT")

        echo ""
        echo "Dependency Check Summary:"
        echo "========================="
        jq -r '.checks | to_entries[] | "\(.key): \(.value.status)"' "$OUTPUT"
        echo ""

        if [[ "$status" == "PASS" ]]; then
            log_info "✓ All dependencies satisfied"
            exit 0
        else
            log_error "✗ Missing or failed dependencies"

            if [[ "$FIX" == true ]]; then
                log_info "Attempting to fix missing dependencies..."
                # Add fix logic here if needed
            fi

            exit 1
        fi
    else
        log_error "Failed to generate dependency report"
        exit 2
    fi
}

main "$@"
