#!/usr/bin/env bash

# validate-data.sh - Comprehensive dataset validation for ML training
# Validates data format, schema, quality, and training readiness

set -euo pipefail

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Default values
DATA_PATH=""
FORMAT="jsonl"
SCHEMA=""
SAMPLE_SIZE=""
CHECK_DUPLICATES=false
CHECK_NULL=false
CHECK_LENGTH=false
CHECK_TOKENS=false
TOKENIZER="gpt2"
MAX_LENGTH=2048
OUTPUT="validation-report.json"
VERBOSE=false
DEBUG=false

# Usage information
usage() {
    cat << EOF
Usage: validate-data.sh [OPTIONS]

Validate training datasets for format compliance, data quality, and schema conformance.

Options:
    --data-path PATH        Path to dataset file or directory (required)
    --format FORMAT         Data format: jsonl, csv, parquet, arrow (default: jsonl)
    --schema PATH           Path to validation schema JSON file
    --sample-size N         Number of samples to validate (default: all)
    --check-duplicates      Check for duplicate entries
    --check-null            Check for null/missing values
    --check-length          Validate text length constraints
    --check-tokens          Validate tokenization compatibility
    --tokenizer MODEL       Tokenizer to use (default: gpt2)
    --max-length N          Maximum sequence length (default: 2048)
    --output REPORT         Output validation report path (default: validation-report.json)
    --verbose               Enable detailed output
    --debug                 Enable debug mode
    -h, --help              Show this help message

Examples:
    # Basic validation
    bash validate-data.sh --data-path ./data/train.jsonl

    # Full validation with schema
    bash validate-data.sh \\
        --data-path ./data/train.jsonl \\
        --schema ./schema.json \\
        --check-duplicates \\
        --check-tokens \\
        --max-length 2048

Exit Codes:
    0 - All validations passed
    1 - Validation errors found
    2 - Script error (invalid arguments, file not found)
EOF
}

# Logging functions
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1" >&2
}

log_debug() {
    if [[ "$DEBUG" == true ]]; then
        echo -e "[DEBUG] $1"
    fi
}

# Check dependencies
check_dependencies() {
    local missing_deps=()

    if ! command -v python3 &> /dev/null; then
        missing_deps+=("python3")
    fi

    if ! command -v jq &> /dev/null; then
        missing_deps+=("jq")
    fi

    if [[ ${#missing_deps[@]} -gt 0 ]]; then
        log_error "Missing required dependencies: ${missing_deps[*]}"
        log_error "Install with: sudo apt-get install ${missing_deps[*]}"
        return 1
    fi

    # Check Python packages
    if [[ "$FORMAT" == "parquet" ]] && ! python3 -c "import pandas" 2>/dev/null; then
        log_error "pandas required for parquet format. Install: pip install pandas pyarrow"
        return 1
    fi

    if [[ "$CHECK_TOKENS" == true ]] && ! python3 -c "import transformers" 2>/dev/null; then
        log_error "transformers required for token validation. Install: pip install transformers"
        return 1
    fi

    return 0
}

# Validate data file exists and is readable
validate_file() {
    if [[ ! -e "$DATA_PATH" ]]; then
        log_error "Data path does not exist: $DATA_PATH"
        return 1
    fi

    if [[ -d "$DATA_PATH" ]]; then
        log_info "Validating directory: $DATA_PATH"
        # Count files in directory
        local file_count=$(find "$DATA_PATH" -type f -name "*.$FORMAT" | wc -l)
        if [[ $file_count -eq 0 ]]; then
            log_error "No .$FORMAT files found in directory"
            return 1
        fi
        log_info "Found $file_count .$FORMAT files"
    elif [[ -f "$DATA_PATH" ]]; then
        log_info "Validating file: $DATA_PATH"
        if [[ ! -r "$DATA_PATH" ]]; then
            log_error "File is not readable: $DATA_PATH"
            return 1
        fi
    else
        log_error "Invalid data path: $DATA_PATH"
        return 1
    fi

    return 0
}

# Validate JSONL format
validate_jsonl() {
    local data_file="$1"
    local validation_output="$2"

    log_info "Validating JSONL format..."

    python3 - "$data_file" "$SCHEMA" "$SAMPLE_SIZE" "$CHECK_DUPLICATES" "$CHECK_NULL" "$CHECK_TOKENS" "$TOKENIZER" "$MAX_LENGTH" "$validation_output" << 'PYTHON_SCRIPT'
import json
import sys
from collections import defaultdict, Counter
import hashlib

data_file = sys.argv[1]
schema_file = sys.argv[2] if sys.argv[2] else None
sample_size = int(sys.argv[3]) if sys.argv[3] else None
check_duplicates = sys.argv[4] == 'true'
check_null = sys.argv[5] == 'true'
check_tokens = sys.argv[6] == 'true'
tokenizer_name = sys.argv[7]
max_length = int(sys.argv[8])
output_file = sys.argv[9]

# Initialize counters
total_samples = 0
valid_samples = 0
errors = []
statistics = defaultdict(list)
seen_hashes = set()
duplicates = []

# Load schema if provided
schema = None
if schema_file:
    try:
        with open(schema_file, 'r') as f:
            schema = json.load(f)
    except Exception as e:
        print(f"Error loading schema: {e}", file=sys.stderr)

# Load tokenizer if needed
tokenizer = None
if check_tokens:
    try:
        from transformers import AutoTokenizer
        tokenizer = AutoTokenizer.from_pretrained(tokenizer_name)
    except Exception as e:
        print(f"Error loading tokenizer: {e}", file=sys.stderr)
        check_tokens = False

# Process file line by line
with open(data_file, 'r', encoding='utf-8') as f:
    for line_num, line in enumerate(f, 1):
        if sample_size and line_num > sample_size:
            break

        total_samples += 1

        # Parse JSON
        try:
            obj = json.loads(line.strip())
        except json.JSONDecodeError as e:
            errors.append({
                "line": line_num,
                "error": f"Invalid JSON: {str(e)}"
            })
            continue

        # Check for duplicates
        if check_duplicates:
            content_hash = hashlib.md5(json.dumps(obj, sort_keys=True).encode()).hexdigest()
            if content_hash in seen_hashes:
                duplicates.append({"line": line_num, "hash": content_hash})
            else:
                seen_hashes.add(content_hash)

        # Schema validation
        if schema:
            required_fields = schema.get('required', [])
            for field in required_fields:
                if field not in obj:
                    errors.append({
                        "line": line_num,
                        "field": field,
                        "error": f"Required field '{field}' missing"
                    })

        # Check for null values
        if check_null:
            for key, value in obj.items():
                if value is None or value == "":
                    errors.append({
                        "line": line_num,
                        "field": key,
                        "error": f"Null or empty value in field '{key}'"
                    })

        # Token validation
        if check_tokens and 'text' in obj:
            text = obj['text']
            if tokenizer:
                try:
                    tokens = tokenizer.encode(text)
                    token_count = len(tokens)
                    statistics['token_counts'].append(token_count)

                    if token_count > max_length:
                        errors.append({
                            "line": line_num,
                            "field": "text",
                            "error": f"Exceeds max token length: {token_count} > {max_length}"
                        })
                except Exception as e:
                    errors.append({
                        "line": line_num,
                        "field": "text",
                        "error": f"Tokenization error: {str(e)}"
                    })

        # Collect statistics
        if 'text' in obj:
            statistics['text_lengths'].append(len(obj['text']))
        if 'label' in obj:
            statistics['labels'].append(obj['label'])

        if not any(e.get('line') == line_num for e in errors):
            valid_samples += 1

# Calculate summary statistics
summary_stats = {}
if statistics['text_lengths']:
    summary_stats['avg_text_length'] = sum(statistics['text_lengths']) / len(statistics['text_lengths'])
    summary_stats['min_text_length'] = min(statistics['text_lengths'])
    summary_stats['max_text_length'] = max(statistics['text_lengths'])

if statistics['token_counts']:
    summary_stats['avg_token_count'] = sum(statistics['token_counts']) / len(statistics['token_counts'])
    summary_stats['min_token_count'] = min(statistics['token_counts'])
    summary_stats['max_token_count'] = max(statistics['token_counts'])

if statistics['labels']:
    label_counts = Counter(statistics['labels'])
    summary_stats['label_distribution'] = dict(label_counts)

# Generate report
report = {
    "status": "PASS" if len(errors) == 0 else "FAIL",
    "total_samples": total_samples,
    "valid_samples": valid_samples,
    "invalid_samples": total_samples - valid_samples,
    "validation_errors": errors[:100],  # Limit to first 100 errors
    "duplicates": duplicates[:50] if check_duplicates else [],
    "statistics": summary_stats,
    "recommendations": []
}

# Add recommendations
if errors:
    report["recommendations"].append(f"Fix {len(errors)} validation errors before training")
if duplicates:
    report["recommendations"].append(f"Found {len(duplicates)} duplicate entries - consider deduplication")
if 'label_distribution' in summary_stats:
    labels = summary_stats['label_distribution']
    if len(labels) > 1:
        max_count = max(labels.values())
        min_count = min(labels.values())
        if max_count / min_count > 3:
            report["recommendations"].append("Label distribution is imbalanced - consider class weighting")

# Save report
with open(output_file, 'w') as f:
    json.dump(report, f, indent=2)

# Print summary
print(f"\nValidation Summary:")
print(f"Total samples: {total_samples}")
print(f"Valid samples: {valid_samples}")
print(f"Invalid samples: {total_samples - valid_samples}")
print(f"Errors found: {len(errors)}")
if duplicates:
    print(f"Duplicates found: {len(duplicates)}")

# Exit with appropriate code
sys.exit(0 if len(errors) == 0 else 1)
PYTHON_SCRIPT
}

# Main validation logic
main() {
    # Parse arguments
    while [[ $# -gt 0 ]]; do
        case $1 in
            --data-path)
                DATA_PATH="$2"
                shift 2
                ;;
            --format)
                FORMAT="$2"
                shift 2
                ;;
            --schema)
                SCHEMA="$2"
                shift 2
                ;;
            --sample-size)
                SAMPLE_SIZE="$2"
                shift 2
                ;;
            --check-duplicates)
                CHECK_DUPLICATES=true
                shift
                ;;
            --check-null)
                CHECK_NULL=true
                shift
                ;;
            --check-length)
                CHECK_LENGTH=true
                shift
                ;;
            --check-tokens)
                CHECK_TOKENS=true
                shift
                ;;
            --tokenizer)
                TOKENIZER="$2"
                shift 2
                ;;
            --max-length)
                MAX_LENGTH="$2"
                shift 2
                ;;
            --output)
                OUTPUT="$2"
                shift 2
                ;;
            --verbose)
                VERBOSE=true
                shift
                ;;
            --debug)
                DEBUG=true
                VERBOSE=true
                shift
                ;;
            -h|--help)
                usage
                exit 0
                ;;
            *)
                log_error "Unknown option: $1"
                usage
                exit 2
                ;;
        esac
    done

    # Validate required arguments
    if [[ -z "$DATA_PATH" ]]; then
        log_error "Missing required argument: --data-path"
        usage
        exit 2
    fi

    # Check dependencies
    if ! check_dependencies; then
        exit 2
    fi

    # Validate file exists
    if ! validate_file; then
        exit 2
    fi

    # Run format-specific validation
    log_info "Starting validation with format: $FORMAT"

    case $FORMAT in
        jsonl)
            if [[ -f "$DATA_PATH" ]]; then
                validate_jsonl "$DATA_PATH" "$OUTPUT"
            else
                # Validate directory of JSONL files
                for file in "$DATA_PATH"/*.jsonl; do
                    if [[ -f "$file" ]]; then
                        log_info "Validating: $file"
                        local file_output="${OUTPUT%.json}_$(basename "$file").json"
                        validate_jsonl "$file" "$file_output"
                    fi
                done
            fi
            ;;
        csv|parquet|arrow)
            log_error "Format '$FORMAT' validation not yet implemented"
            exit 2
            ;;
        *)
            log_error "Unknown format: $FORMAT"
            exit 2
            ;;
    esac

    # Display results
    if [[ -f "$OUTPUT" ]]; then
        log_info "Validation report saved to: $OUTPUT"

        if command -v jq &> /dev/null; then
            local status=$(jq -r '.status' "$OUTPUT")
            if [[ "$status" == "PASS" ]]; then
                log_info "✓ Validation PASSED"
                exit 0
            else
                log_error "✗ Validation FAILED"
                if [[ "$VERBOSE" == true ]]; then
                    jq '.' "$OUTPUT"
                fi
                exit 1
            fi
        fi
    else
        log_error "Validation report not generated"
        exit 2
    fi
}

# Run main function
main "$@"
