#!/usr/bin/env bash

# validate-model.sh - Model checkpoint and configuration validation
# Verifies model integrity, loadability, and inference readiness

set -euo pipefail

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# Default values
MODEL_PATH=""
FRAMEWORK="pytorch"
CHECK_WEIGHTS=false
CHECK_CONFIG=false
CHECK_TOKENIZER=false
CHECK_INFERENCE=false
SAMPLE_INPUT="Hello, world!"
EXPECTED_OUTPUT=""
OUTPUT="model-validation-report.json"
VERBOSE=false

# Usage
usage() {
    cat << EOF
Usage: validate-model.sh [OPTIONS]

Validate model checkpoints, configurations, and inference readiness.

Options:
    --model-path PATH       Path to model checkpoint directory (required)
    --framework FRAMEWORK   Framework: pytorch, tensorflow, jax (default: pytorch)
    --check-weights         Verify model weights are loadable
    --check-config          Validate model configuration file
    --check-tokenizer       Verify tokenizer files are present
    --check-inference       Test inference with sample input
    --sample-input TEXT     Sample text for inference test
    --expected-output TEXT  Expected output for verification
    --output REPORT         Output validation report path
    --verbose               Enable detailed output
    -h, --help              Show this help message

Examples:
    # Basic validation
    bash validate-model.sh --model-path ./checkpoints/final

    # Full validation with inference test
    bash validate-model.sh \\
        --model-path ./checkpoints/final \\
        --framework pytorch \\
        --check-weights \\
        --check-config \\
        --check-tokenizer \\
        --check-inference

Exit Codes:
    0 - All validations passed
    1 - Validation errors found
    2 - Script error
EOF
}

# Logging
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1" >&2
}

# Check dependencies
check_dependencies() {
    if ! command -v python3 &> /dev/null; then
        log_error "python3 is required"
        return 1
    fi

    if [[ "$FRAMEWORK" == "pytorch" ]]; then
        if ! python3 -c "import torch" 2>/dev/null; then
            log_error "PyTorch not installed. Install: pip install torch"
            return 1
        fi
    elif [[ "$FRAMEWORK" == "tensorflow" ]]; then
        if ! python3 -c "import tensorflow" 2>/dev/null; then
            log_error "TensorFlow not installed. Install: pip install tensorflow"
            return 1
        fi
    fi

    return 0
}

# Validate model directory structure
validate_structure() {
    local model_path="$1"

    if [[ ! -d "$model_path" ]]; then
        log_error "Model path is not a directory: $model_path"
        return 1
    fi

    log_info "Checking model directory structure..."

    # Check for common model files
    local has_files=false

    if [[ "$FRAMEWORK" == "pytorch" ]]; then
        if [[ -f "$model_path/pytorch_model.bin" ]] || [[ -f "$model_path/model.safetensors" ]] || ls "$model_path"/*.pth &>/dev/null; then
            has_files=true
        fi
    fi

    if [[ "$has_files" == false ]]; then
        log_warn "No model weight files found in $model_path"
        return 1
    fi

    return 0
}

# Validate model using Python
validate_model_python() {
    local model_path="$1"
    local framework="$2"
    local check_weights="$3"
    local check_config="$4"
    local check_tokenizer="$5"
    local check_inference="$6"
    local sample_input="$7"
    local output_file="$8"

    python3 - "$model_path" "$framework" "$check_weights" "$check_config" "$check_tokenizer" "$check_inference" "$sample_input" "$output_file" << 'PYTHON_SCRIPT'
import json
import sys
import os
from pathlib import Path
import time

model_path = Path(sys.argv[1])
framework = sys.argv[2]
check_weights = sys.argv[3] == 'true'
check_config = sys.argv[4] == 'true'
check_tokenizer = sys.argv[5] == 'true'
check_inference = sys.argv[6] == 'true'
sample_input = sys.argv[7]
output_file = sys.argv[8]

report = {
    "status": "PASS",
    "model_path": str(model_path),
    "framework": framework,
    "checks": {},
    "model_info": {},
    "errors": []
}

# File structure check
print("Checking file structure...")
files = list(model_path.glob("*"))
report["checks"]["file_structure"] = "PASS" if files else "FAIL"
if not files:
    report["errors"].append("No files found in model directory")
    report["status"] = "FAIL"

# Config check
if check_config:
    print("Checking model configuration...")
    config_file = model_path / "config.json"
    if config_file.exists():
        try:
            with open(config_file, 'r') as f:
                config = json.load(f)
            report["checks"]["config"] = "PASS"
            report["model_info"]["architecture"] = config.get("architectures", ["unknown"])[0]
            report["model_info"]["model_type"] = config.get("model_type", "unknown")
        except Exception as e:
            report["checks"]["config"] = "FAIL"
            report["errors"].append(f"Config validation error: {str(e)}")
            report["status"] = "FAIL"
    else:
        report["checks"]["config"] = "FAIL"
        report["errors"].append("config.json not found")
        report["status"] = "FAIL"

# Weights check
if check_weights and framework == "pytorch":
    print("Checking model weights...")
    try:
        import torch

        # Try loading weights
        weight_files = list(model_path.glob("*.bin")) + list(model_path.glob("*.safetensors")) + list(model_path.glob("*.pth"))

        if weight_files:
            try:
                # Try to load from pretrained
                from transformers import AutoModel
                model = AutoModel.from_pretrained(str(model_path))

                # Count parameters
                total_params = sum(p.numel() for p in model.parameters())
                report["model_info"]["parameters"] = f"{total_params / 1e9:.1f}B" if total_params > 1e9 else f"{total_params / 1e6:.1f}M"

                # Check for LoRA
                adapter_config = model_path / "adapter_config.json"
                if adapter_config.exists():
                    with open(adapter_config, 'r') as f:
                        adapter_cfg = json.load(f)
                    report["model_info"]["lora_enabled"] = True
                    report["model_info"]["lora_rank"] = adapter_cfg.get("r", "unknown")
                else:
                    report["model_info"]["lora_enabled"] = False

                report["checks"]["weights"] = "PASS"
            except Exception as e:
                report["checks"]["weights"] = "FAIL"
                report["errors"].append(f"Weight loading error: {str(e)}")
                report["status"] = "FAIL"
        else:
            report["checks"]["weights"] = "FAIL"
            report["errors"].append("No weight files found")
            report["status"] = "FAIL"

    except ImportError:
        report["checks"]["weights"] = "SKIP"
        report["errors"].append("PyTorch not available for weight validation")

# Tokenizer check
if check_tokenizer:
    print("Checking tokenizer files...")
    tokenizer_files = ["tokenizer.json", "tokenizer_config.json", "vocab.json", "special_tokens_map.json"]
    found_files = [f for f in tokenizer_files if (model_path / f).exists()]

    if found_files:
        try:
            from transformers import AutoTokenizer
            tokenizer = AutoTokenizer.from_pretrained(str(model_path))
            report["checks"]["tokenizer"] = "PASS"
            report["model_info"]["vocab_size"] = tokenizer.vocab_size
        except Exception as e:
            report["checks"]["tokenizer"] = "FAIL"
            report["errors"].append(f"Tokenizer loading error: {str(e)}")
            report["status"] = "FAIL"
    else:
        report["checks"]["tokenizer"] = "FAIL"
        report["errors"].append("Tokenizer files not found")
        report["status"] = "FAIL"

# Inference check
if check_inference and framework == "pytorch":
    print("Running inference test...")
    try:
        from transformers import AutoTokenizer, AutoModelForCausalLM
        import torch

        tokenizer = AutoTokenizer.from_pretrained(str(model_path))
        model = AutoModelForCausalLM.from_pretrained(str(model_path))

        # Run inference
        start_time = time.time()
        inputs = tokenizer(sample_input, return_tensors="pt")
        with torch.no_grad():
            outputs = model.generate(**inputs, max_new_tokens=20)
        output_text = tokenizer.decode(outputs[0], skip_special_tokens=True)
        latency_ms = int((time.time() - start_time) * 1000)

        report["checks"]["inference"] = "PASS"
        report["inference_test"] = {
            "input": sample_input,
            "output": output_text,
            "latency_ms": latency_ms
        }

        # Estimate memory
        model_size_mb = sum(p.numel() * p.element_size() for p in model.parameters()) / (1024 ** 2)
        report["memory_estimate"] = {
            "model_size_gb": round(model_size_mb / 1024, 1),
            "inference_vram_gb": round(model_size_mb * 1.2 / 1024, 1),  # Rough estimate
            "training_vram_gb": round(model_size_mb * 2.5 / 1024, 1)  # Rough estimate with gradients
        }

    except Exception as e:
        report["checks"]["inference"] = "FAIL"
        report["errors"].append(f"Inference test error: {str(e)}")
        report["status"] = "FAIL"

# Save report
with open(output_file, 'w') as f:
    json.dump(report, f, indent=2)

# Print summary
print(f"\nValidation Summary:")
print(f"Status: {report['status']}")
for check, status in report["checks"].items():
    print(f"  {check}: {status}")

if report["errors"]:
    print(f"\nErrors found: {len(report['errors'])}")
    for error in report["errors"]:
        print(f"  - {error}")

sys.exit(0 if report["status"] == "PASS" else 1)
PYTHON_SCRIPT
}

# Main
main() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --model-path)
                MODEL_PATH="$2"
                shift 2
                ;;
            --framework)
                FRAMEWORK="$2"
                shift 2
                ;;
            --check-weights)
                CHECK_WEIGHTS=true
                shift
                ;;
            --check-config)
                CHECK_CONFIG=true
                shift
                ;;
            --check-tokenizer)
                CHECK_TOKENIZER=true
                shift
                ;;
            --check-inference)
                CHECK_INFERENCE=true
                shift
                ;;
            --sample-input)
                SAMPLE_INPUT="$2"
                shift 2
                ;;
            --expected-output)
                EXPECTED_OUTPUT="$2"
                shift 2
                ;;
            --output)
                OUTPUT="$2"
                shift 2
                ;;
            --verbose)
                VERBOSE=true
                shift
                ;;
            -h|--help)
                usage
                exit 0
                ;;
            *)
                log_error "Unknown option: $1"
                usage
                exit 2
                ;;
        esac
    done

    if [[ -z "$MODEL_PATH" ]]; then
        log_error "Missing required argument: --model-path"
        usage
        exit 2
    fi

    if ! check_dependencies; then
        exit 2
    fi

    if ! validate_structure "$MODEL_PATH"; then
        exit 1
    fi

    validate_model_python "$MODEL_PATH" "$FRAMEWORK" "$CHECK_WEIGHTS" "$CHECK_CONFIG" "$CHECK_TOKENIZER" "$CHECK_INFERENCE" "$SAMPLE_INPUT" "$OUTPUT"

    if [[ -f "$OUTPUT" ]]; then
        log_info "Validation report saved to: $OUTPUT"

        if command -v jq &> /dev/null; then
            local status=$(jq -r '.status' "$OUTPUT")
            if [[ "$status" == "PASS" ]]; then
                log_info "✓ Model validation PASSED"
                if [[ "$VERBOSE" == true ]]; then
                    jq '.' "$OUTPUT"
                fi
                exit 0
            else
                log_error "✗ Model validation FAILED"
                jq '.' "$OUTPUT"
                exit 1
            fi
        fi
    fi
}

main "$@"
