#!/bin/bash
# Setup Supabase CMS in an Astro project

set -e

echo "🚀 Setting up Supabase CMS for Astro..."

# Check if we're in an Astro project
if [ ! -f "package.json" ]; then
  echo "❌ Error: package.json not found. Are you in an Astro project?"
  exit 1
fi

if ! grep -q "astro" package.json; then
  echo "❌ Error: This doesn't appear to be an Astro project"
  exit 1
fi

# Install Supabase dependencies
echo "📦 Installing Supabase dependencies..."
npm install @supabase/supabase-js

# Create Supabase client directory
echo "📁 Creating Supabase client configuration..."
mkdir -p src/lib/supabase

# Create Supabase client file
cat > src/lib/supabase/client.ts << 'EOF'
import { createClient } from '@supabase/supabase-js';
import type { Database } from './database.types';

const supabaseUrl = import.meta.env.PUBLIC_SUPABASE_URL;
const supabaseAnonKey = import.meta.env.PUBLIC_SUPABASE_ANON_KEY;

if (!supabaseUrl || !supabaseAnonKey) {
  throw new Error('Missing Supabase environment variables');
}

export const supabase = createClient<Database>(supabaseUrl, supabaseAnonKey, {
  auth: {
    persistSession: false,
    autoRefreshToken: false,
    detectSessionInUrl: false
  }
});
EOF

# Create .env template
if [ ! -f ".env" ]; then
  echo "📝 Creating .env template..."
  cat > .env << 'EOF'
PUBLIC_SUPABASE_URL=your-project-url
PUBLIC_SUPABASE_ANON_KEY=your-anon-key
EOF
  echo "⚠️  Please update .env with your Supabase credentials"
fi

# Add .env to .gitignore if not already there
if [ -f ".gitignore" ]; then
  if ! grep -q "^.env$" .gitignore; then
    echo ".env" >> .gitignore
  fi
fi

# Create database types placeholder
cat > src/lib/supabase/database.types.ts << 'EOF'
// This file will be generated by Supabase CLI
// Run: npm run generate:types

export type Json =
  | string
  | number
  | boolean
  | null
  | { [key: string]: Json | undefined }
  | Json[]

export interface Database {
  public: {
    Tables: {
      // Your tables will be generated here
    }
    Views: {
      [_ in never]: never
    }
    Functions: {
      [_ in never]: never
    }
    Enums: {
      [_ in never]: never
    }
  }
}
EOF

# Add scripts to package.json
echo "📝 Adding npm scripts..."
if command -v node > /dev/null; then
  node << 'NODESCRIPT'
const fs = require('fs');
const packageJson = JSON.parse(fs.readFileSync('package.json', 'utf8'));

if (!packageJson.scripts) {
  packageJson.scripts = {};
}

packageJson.scripts['generate:types'] = 'supabase gen types typescript --project-id $PROJECT_ID > src/lib/supabase/database.types.ts';

fs.writeFileSync('package.json', JSON.stringify(packageJson, null, 2) + '\n');
NODESCRIPT
fi

echo "✅ Supabase CMS setup complete!"
echo ""
echo "Next steps:"
echo "1. Update .env with your Supabase project credentials"
echo "2. Apply database migrations: ./scripts/apply-migration.sh"
echo "3. Generate TypeScript types: npm run generate:types"
echo "4. Apply RLS policies: ./scripts/apply-rls-policies.sh"
