#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
HWP 5.x to TXT Converter (v0.1.0)

Entry point for HWP binary format conversion.
"""

import os
import sys
from .ole import OleReader
from .parser import read_hwp_metadata, extract_content_from_section, Table


def convert_to_text(content_list):
    """Convert content list to plain text."""
    result = []
    table_count = 1
    
    for item in content_list:
        if isinstance(item, Table):
            result.append(f"\n\n[표 {table_count}]")
            result.append(item.to_text())
            table_count += 1
        else:
            result.append(item)
            
    return "\n\n".join(result)


def convert_to_html(content_list):
    """Convert content list to HTML."""
    html = ["<!DOCTYPE html>", "<html>", "<head>", '<meta charset="utf-8">', "</head>", "<body>"]
    
    for item in content_list:
        if isinstance(item, Table):
            html.append(item.to_html())
        else:
            # Simple paragraph wrapping
            escaped_text = item.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;")
            html.append(f"<p>{escaped_text}</p>")
            
    html.append("</body>")
    html.append("</html>")
    return "\n".join(html)


def convert_to_markdown(content_list):
    """Convert content list to Markdown."""
    md = []
    
    for item in content_list:
        if isinstance(item, Table):
            md.append("\n" + item.to_markdown() + "\n")
        else:
            md.append(item + "\n")
            
    return "\n".join(md)


def extract_content_from_hwp(hwp_path, output_format="txt"):
    """
    Extract content from HWP file in specified format.
    
    Args:
        hwp_path: Path to HWP file
        output_format: 'txt', 'html', or 'md'
        
    Returns:
        Converted content as string, or error message
    """
    if not os.path.exists(hwp_path):
        return f"[Error] File not found: {hwp_path}"
    
    try:
        with OleReader(hwp_path) as ole:
            meta = read_hwp_metadata(ole)
            print(f"Version: {meta['version']}, Compressed: {meta['compressed']}")
            
            all_content = []
            idx = 0
            
            # 모든 섹션 순회
            while True:
                name = f"BodyText/Section{idx}"
                if name not in ole.list_streams():
                    break
                
                print(f"📄 Processing {name}...")
                section_content = extract_content_from_section(ole, name)
                all_content.extend(section_content)
                idx += 1
            
            # 포맷에 따른 변환
            if output_format == "html":
                return convert_to_html(all_content)
            elif output_format == "md":
                return convert_to_markdown(all_content)
            else:
                return convert_to_text(all_content)
    
    except Exception as e:
        return f"[Error] Failed: {e}"


def convert_hwp(input_path, output_path="/tmp/converted_text.txt", output_format="txt"):
    """
    Convert HWP file to plain text, HTML, or Markdown.
    
    Args:
        input_path: Path to input HWP file
        output_path: Path to output file
        output_format: 'txt', 'html', or 'md'
        
    Returns:
        True if conversion succeeded, False otherwise
    """
    if not os.path.exists(input_path):
        print(f"❌ File not found: {input_path}")
        return False
    
    print(f"🔧 Extracting content as {output_format.upper()}...")
    text = extract_content_from_hwp(input_path, output_format)
    
    if text.startswith("[Error]") or not text.strip():
        print(text)
        return False
    
    try:
        with open(output_path, "w", encoding="utf-8") as f:
            f.write(text.strip())
        print(f"✅ Output saved: {output_path}")
        return True
    except Exception as e:
        print(f"❌ Save failed: {e}")
        return False


def main():
    """Command-line interface entry point"""
    if len(sys.argv) < 2:
        print("Usage: python -m scripts.hwp5.converter <file.hwp> [output_path] [--format txt|html|md]")
        sys.exit(1)
    
    hwp_file = sys.argv[1]
    
    output_path = None
    output_format = "txt"
    
    # Parse arguments manually to avoid imports
    args = sys.argv[2:]
    if args and not args[0].startswith("--"):
        output_path = args[0]
        args = args[1:]
        
    for i, arg in enumerate(args):
        if arg == "--format" and i + 1 < len(args):
            output_format = args[i+1]
            
    if not output_path:
        ext = "txt"
        if output_format == "html": ext = "html"
        elif output_format == "md": ext = "md"
        output_path = f"/tmp/converted.{ext}"
    
    output_dir = os.path.dirname(output_path)
    if output_dir:
        os.makedirs(output_dir, exist_ok=True)
    
    success = convert_hwp(hwp_file, output_path, output_format)
    sys.exit(0 if success else 1)


if __name__ == "__main__":
    main()
