#!/usr/bin/env python3
"""
HWP 파일에서 이미지를 추출하고 BMP를 PNG로 변환하는 스크립트
"""

import olefile
import os
import zlib
import sys
from io import BytesIO

# PIL is optional but recommended for BMP conversion
try:
    from PIL import Image
    HAS_PIL = True
except ImportError:
    HAS_PIL = False


def extract_and_convert_images(hwp_path, output_dir, convert_to_png=True):
    """
    HWP 파일에서 이미지를 추출하고 선택적으로 PNG로 변환합니다.
    
    Args:
        hwp_path (str): HWP 파일 경로
        output_dir (str): 이미지를 저장할 디렉토리
        convert_to_png (bool): BMP를 PNG로 변환할지 여부
    
    Returns:
        list: 추출된 이미지 파일 경로 목록
    """
    os.makedirs(output_dir, exist_ok=True)
    extracted_files = []
    
    try:
        ole = olefile.OleFileIO(hwp_path)
        bindata_streams = [s for s in ole.listdir() if s[0] == 'BinData']
        
        if not bindata_streams:
            print("⚠ BinData 폴더를 찾을 수 없습니다.")
            ole.close()
            return extracted_files
        
        print(f"📦 {len(bindata_streams)}개의 이미지를 처리합니다...")
        
        for idx, entry in enumerate(bindata_streams, 1):
            filename = entry[-1]
            stream = ole.openstream(entry)
            data = stream.read()
            
            # zlib 압축 해제
            try:
                data = zlib.decompress(data, -15)
            except zlib.error:
                pass
            
            # 파일 형식 확인
            ext = detect_image_format(data)
            
            # BMP를 PNG로 변환
            if convert_to_png and ext == '.bmp' and HAS_PIL:
                try:
                    img = Image.open(BytesIO(data))
                    output_path = os.path.join(output_dir, f"image_{idx:02d}.png")
                    img.save(output_path, 'PNG')
                    print(f"  🖼️  변환: {filename} → image_{idx:02d}.png")
                except Exception as e:
                    print(f"  ⚠ PNG 변환 실패: {e}")
                    output_path = os.path.join(output_dir, f"image_{idx:02d}{ext}")
                    with open(output_path, 'wb') as f:
                        f.write(data)
            else:
                output_path = os.path.join(output_dir, f"image_{idx:02d}{ext}")
                with open(output_path, 'wb') as f:
                    f.write(data)
                print(f"  💾 저장: image_{idx:02d}{ext}")
            
            extracted_files.append(output_path)
        
        ole.close()
        print(f"\n✅ 총 {len(extracted_files)}개의 이미지를 추출했습니다.")
        
    except Exception as e:
        print(f"❌ 오류 발생: {e}")
        return []
    
    return extracted_files


def detect_image_format(data):
    """이미지 형식 감지"""
    if len(data) < 4:
        return '.bin'
    
    if data[:4] == b'\x89PNG':
        return '.png'
    elif data[:2] == b'\xff\xd8':
        return '.jpg'
    elif data[:2] == b'BM':
        return '.bmp'
    elif data[:3] == b'GIF':
        return '.gif'
    elif data[:2] in (b'II', b'MM'):
        return '.tiff'
    else:
        return '.bin'


def main():
    if len(sys.argv) < 2:
        print("사용법: python -m scripts.hwp5.extract_images <hwp_file> [output_dir]")
        sys.exit(1)
    
    hwp_path = sys.argv[1]
    output_dir = sys.argv[2] if len(sys.argv) > 2 else './extracted_images'
    
    if not os.path.exists(hwp_path):
        print(f"❌ 파일을 찾을 수 없습니다: {hwp_path}")
        sys.exit(1)
    
    if not HAS_PIL:
        print("⚠ Pillow가 설치되지 않아 BMP 변환을 건너뜁니다.")
        print("  설치: pip install Pillow")
        
    extract_and_convert_images(hwp_path, output_dir, convert_to_png=True)


if __name__ == '__main__':
    main()
