"""
微博热搜产品创意分析 - 命令配置管理
处理斜杠命令的参数解析、验证和配置转换
"""
from dataclasses import dataclass, field
from typing import List, Tuple, Optional, Dict, Any
import os


@dataclass
class CommandConfig:
    """命令配置数据类"""
    max_topics: int = 10
    output_dir: str = "./output/reports"
    industry_focus: List[str] = field(default_factory=lambda: ["all"])
    scoring_weights: Tuple[int, int] = (80, 20)
    ideas_per_topic: int = 3
    quiet_mode: bool = False
    focus_string: str = "all"  # 保存原始focus字符串用于显示
    weights_string: str = "80,20"  # 保存原始权重字符串用于显示

    def __post_init__(self):
        """初始化后处理"""
        if isinstance(self.industry_focus, str):
            self.industry_focus = [self.industry_focus]

    def validate(self) -> List[str]:
        """验证配置，返回错误列表"""
        errors = []

        # 验证话题数量
        if not isinstance(self.max_topics, int) or not 5 <= self.max_topics <= 20:
            errors.append("话题数量必须是5-20之间的整数")

        # 验证创意数量
        if not isinstance(self.ideas_per_topic, int) or not 1 <= self.ideas_per_topic <= 5:
            errors.append("每个话题的创意数必须是1-5之间的整数")

        # 验证权重
        if (not isinstance(self.scoring_weights, tuple) or
            len(self.scoring_weights) != 2 or
            not all(isinstance(w, int) for w in self.scoring_weights) or
            sum(self.scoring_weights) != 100 or
            not all(w >= 0 for w in self.scoring_weights)):
            errors.append("权重必须是两个非负整数，且和为100")

        # 验证输出目录
        if not isinstance(self.output_dir, str) or not self.output_dir.strip():
            errors.append("输出目录不能为空")

        # 验证行业领域
        valid_industries = ['tech', 'lifestyle', 'finance', 'entertainment', 'education', 'all']
        invalid_industries = [ind for ind in self.industry_focus if ind not in valid_industries]
        if invalid_industries:
            errors.append(f"无效的行业领域: {', '.join(invalid_industries)}")

        return errors

    def to_dict(self) -> Dict[str, Any]:
        """转换为字典格式"""
        return {
            'max_topics': self.max_topics,
            'output_dir': self.output_dir,
            'industry_focus': self.industry_focus,
            'scoring_weights': self.scoring_weights,
            'ideas_per_topic': self.ideas_per_topic,
            'quiet_mode': self.quiet_mode,
            'focus_string': self.focus_string,
            'weights_string': self.weights_string
        }

    def to_display_dict(self) -> Dict[str, str]:
        """转换为显示用的字典（字符串格式）"""
        return {
            'topics': str(self.max_topics),
            'output': self.output_dir,
            'focus': self.focus_string,
            'weights': self.weights_string,
            'ideas': str(self.ideas_per_topic),
            'quiet': str(self.quiet_mode).lower()
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'CommandConfig':
        """从字典创建配置"""
        return cls(
            max_topics=data.get('max_topics', 10),
            output_dir=data.get('output_dir', './output/reports'),
            industry_focus=data.get('industry_focus', ['all']),
            scoring_weights=tuple(data.get('scoring_weights', (80, 20))),
            ideas_per_topic=data.get('ideas_per_topic', 3),
            quiet_mode=data.get('quiet_mode', False),
            focus_string=data.get('focus_string', 'all'),
            weights_string=data.get('weights_string', '80,20')
        )


class ConfigParser:
    """配置解析器"""

    VALID_INDUSTRIES = ['tech', 'lifestyle', 'finance', 'entertainment', 'education', 'all']

    @classmethod
    def parse_command_args(cls, args: Dict[str, Any]) -> CommandConfig:
        """解析命令行参数"""
        config = CommandConfig()

        try:
            # 解析topics参数
            if 'topics' in args:
                config.max_topics = cls._parse_integer(args['topics'], "话题数量", 5, 20)

            # 解析output参数
            if 'output' in args:
                config.output_dir = cls._parse_string(args['output'], "输出目录").strip()

            # 解析focus参数
            if 'focus' in args:
                focus_str = cls._parse_string(args['focus'], "专注领域").lower()
                if focus_str != 'all':
                    industries = [f.strip().lower() for f in focus_str.split(',')]
                    cls._validate_industries(industries)
                    config.industry_focus = industries
                config.focus_string = args['focus']

            # 解析weights参数
            if 'weights' in args:
                weights_str = cls._parse_string(args['weights'], "权重参数")
                weights = cls._parse_weights(weights_str)
                config.scoring_weights = tuple(weights)
                config.weights_string = args['weights']

            # 解析ideas参数
            if 'ideas' in args:
                config.ideas_per_topic = cls._parse_integer(args['ideas'], "创意数量", 1, 5)

            # 解析quiet参数
            if 'quiet' in args:
                config.quiet_mode = cls._parse_boolean(args['quiet'], "静默模式")

            # 验证最终配置
            errors = config.validate()
            if errors:
                raise ValueError("配置验证失败: " + "; ".join(errors))

            return config

        except ValueError as e:
            raise ValueError(f"参数解析失败: {e}")
        except Exception as e:
            raise ValueError(f"解析过程中发生未知错误: {e}")

    @staticmethod
    def _parse_integer(value: Any, name: str, min_val: int = None, max_val: int = None) -> int:
        """解析整数参数"""
        try:
            int_val = int(value)
            if min_val is not None and int_val < min_val:
                raise ValueError(f"{name}不能小于{min_val}")
            if max_val is not None and int_val > max_val:
                raise ValueError(f"{name}不能大于{max_val}")
            return int_val
        except (ValueError, TypeError):
            raise ValueError(f"{name}必须是整数")

    @staticmethod
    def _parse_string(value: Any, name: str) -> str:
        """解析字符串参数"""
        if value is None:
            raise ValueError(f"{name}不能为空")
        return str(value).strip()

    @staticmethod
    def _parse_boolean(value: Any, name: str) -> bool:
        """解析布尔参数"""
        if isinstance(value, bool):
            return value
        if isinstance(value, str):
            return value.lower() in ('true', '1', 'yes', 'on')
        if isinstance(value, (int, float)):
            return bool(value)
        raise ValueError(f"{name}必须是布尔值")

    @staticmethod
    def _parse_weights(weights_str: str) -> List[int]:
        """解析权重参数"""
        try:
            weights = [int(w.strip()) for w in weights_str.split(',')]
            if len(weights) != 2:
                raise ValueError("权重参数必须包含两个值")

            total = sum(weights)
            if total != 100:
                raise ValueError(f"权重之和必须为100，当前为{total}")

            if any(w < 0 for w in weights):
                raise ValueError("权重不能为负数")

            return weights
        except ValueError as e:
            if "invalid literal" in str(e):
                raise ValueError("权重参数格式错误，应为逗号分隔的两个整数")
            raise e

    @classmethod
    def _validate_industries(cls, industries: List[str]) -> None:
        """验证行业领域"""
        invalid = [ind for ind in industries if ind not in cls.VALID_INDUSTRIES]
        if invalid:
            raise ValueError(f"无效的行业领域: {', '.join(invalid)}. "
                           f"有效选项: {', '.join(cls.VALID_INDUSTRIES)}")


class ConfigValidator:
    """配置验证器"""

    @staticmethod
    def validate_output_directory(output_dir: str) -> str:
        """验证并标准化输出目录路径"""
        if not output_dir:
            raise ValueError("输出目录不能为空")

        # 标准化路径
        output_dir = os.path.normpath(output_dir)

        # 如果是相对路径，转换为绝对路径
        if not os.path.isabs(output_dir):
            output_dir = os.path.abspath(output_dir)

        # 确保目录存在或可以创建
        try:
            os.makedirs(output_dir, exist_ok=True)
        except OSError as e:
            raise ValueError(f"无法创建输出目录 {output_dir}: {e}")

        # 检查目录是否可写
        if not os.access(output_dir, os.W_OK):
            raise ValueError(f"输出目录 {output_dir} 不可写")

        return output_dir

    @staticmethod
    def validate_scoring_weights(weights: Tuple[int, int]) -> Tuple[int, int]:
        """验证评分权重"""
        if not isinstance(weights, tuple) or len(weights) != 2:
            raise ValueError("权重必须是包含两个整数的元组")

        interestingness, usefulness = weights

        if not (isinstance(interestingness, int) and isinstance(usefulness, int)):
            raise ValueError("权重必须是整数")

        if not (interestingness + usefulness == 100):
            raise ValueError("权重之和必须为100")

        if not (interestingness >= 0 and usefulness >= 0):
            raise ValueError("权重不能为负数")

        return weights

    @staticmethod
    def get_default_config() -> CommandConfig:
        """获取默认配置"""
        return CommandConfig()


class ConfigMerger:
    """配置合并器"""

    @staticmethod
    def merge_configs(base_config: CommandConfig, override_config: Dict[str, Any]) -> CommandConfig:
        """合并配置"""
        merged_dict = base_config.to_dict()

        # 应用覆盖配置
        for key, value in override_config.items():
            if key in merged_dict:
                merged_dict[key] = value

        # 创建新的配置对象
        return CommandConfig.from_dict(merged_dict)

    @staticmethod
    def apply_environment_overrides(config: CommandConfig) -> CommandConfig:
        """应用环境变量覆盖"""
        env_overrides = {}

        # 检查环境变量
        env_mappings = {
            'WEIBO_TOPICS': ('max_topics', int),
            'WEIBO_OUTPUT_DIR': ('output_dir', str),
            'WEIBO_FOCUS': ('industry_focus', str),
            'WEIBO_WEIGHTS': ('scoring_weights', str),
            'WEIBO_IDEAS': ('ideas_per_topic', int),
            'WEIBO_QUIET': ('quiet_mode', bool)
        }

        for env_var, (config_key, type_func) in env_mappings.items():
            env_value = os.getenv(env_var)
            if env_value is not None:
                try:
                    if type_func == str and config_key == 'industry_focus':
                        # 特殊处理行业领域
                        env_overrides[config_key] = [f.strip().lower() for f in env_value.split(',')]
                    elif type_func == str and config_key == 'scoring_weights':
                        # 特殊处理权重
                        weights = [int(w.strip()) for w in env_value.split(',')]
                        env_overrides[config_key] = tuple(weights)
                    else:
                        env_overrides[config_key] = type_func(env_value)
                except (ValueError, TypeError) as e:
                    print(f"警告: 环境变量 {env_var} 解析失败: {e}")

        if env_overrides:
            return ConfigMerger.merge_configs(config, env_overrides)

        return config


def load_config_from_file(config_file: str) -> Optional[CommandConfig]:
    """从文件加载配置"""
    if not os.path.exists(config_file):
        return None

    try:
        import json
        with open(config_file, 'r', encoding='utf-8') as f:
            data = json.load(f)
        return CommandConfig.from_dict(data)
    except Exception as e:
        print(f"警告: 配置文件 {config_file} 加载失败: {e}")
        return None


def save_config_to_file(config: CommandConfig, config_file: str) -> bool:
    """保存配置到文件"""
    try:
        import json
        os.makedirs(os.path.dirname(config_file), exist_ok=True)
        with open(config_file, 'w', encoding='utf-8') as f:
            json.dump(config.to_dict(), f, ensure_ascii=False, indent=2)
        return True
    except Exception as e:
        print(f"错误: 配置文件 {config_file} 保存失败: {e}")
        return False


# 示例用法和测试
if __name__ == "__main__":
    # 测试配置解析
    test_args = {
        'topics': 15,
        'output': './test_reports',
        'focus': 'tech,lifestyle',
        'weights': '70,30',
        'ideas': 4,
        'quiet': True
    }

    try:
        config = ConfigParser.parse_command_args(test_args)
        print("配置解析成功:")
        print(f"  话题数量: {config.max_topics}")
        print(f"  输出目录: {config.output_dir}")
        print(f"  专注领域: {config.industry_focus}")
        print(f"  评分权重: {config.scoring_weights}")
        print(f"  创意数量: {config.ideas_per_topic}")
        print(f"  静默模式: {config.quiet_mode}")

        # 测试验证
        errors = config.validate()
        if errors:
            print("配置验证失败:")
            for error in errors:
                print(f"  - {error}")
        else:
            print("配置验证通过")

    except Exception as e:
        print(f"测试失败: {e}")