"""
微博热搜产品创意分析主程序
整合所有功能模块，执行完整的分析流程
支持命令行接口和斜杠命令集成
"""
import asyncio
import sys
import os
import argparse
from typing import Dict, List, Optional

# 添加src目录到Python路径
sys.path.append(os.path.join(os.path.dirname(__file__), 'src'))

from src.core.tianapi_scraper import get_weibo_hotsearch, test_api_connection
from src.core.news_analyzer import analyze_topic_background
from src.core.idea_generator import generate_product_ideas
from src.core.scorer import calculate_product_idea_score
from src.utils.html_generator import generate_analysis_report
from src.config.api_config import ANALYSIS_CONFIG

# 导入配置管理模块
try:
    from command_config import CommandConfig, ConfigParser, ConfigValidator, ConfigMerger
except ImportError:
    # 如果在同目录下找不到，尝试从相对路径导入
    sys.path.append(os.path.dirname(__file__))
    from command_config import CommandConfig, ConfigParser, ConfigValidator, ConfigMerger


class WeiboHotSearchAnalyzer:
    """微博热搜产品创意分析器"""

    def __init__(self, config: CommandConfig = None):
        if config:
            self.config = config
            self.max_topics = config.max_topics
            self.num_ideas_per_topic = config.ideas_per_topic
        else:
            self.config = CommandConfig()
            self.max_topics = ANALYSIS_CONFIG['max_topics_to_analyze']
            self.num_ideas_per_topic = 3

    async def run_analysis(self, max_topics: int = None, web_search_func=None, config: CommandConfig = None) -> Dict:
        """
        执行完整的分析流程

        Args:
            max_topics: 最大分析话题数量
            web_search_func: Web搜索函数

        Returns:
            分析结果
        """
        try:
            print("开始微博热搜产品创意分析...")

            # 1. 测试API连接
            if not test_api_connection():
                raise Exception("天行数据API连接失败，请检查网络和API密钥")

            # 2. 获取微博热搜数据
            print("📡 正在获取微博热搜数据...")
            if max_topics:
                hot_topics = get_weibo_hotsearch(max_topics)
            else:
                hot_topics = get_weibo_hotsearch(self.max_topics)

            if not hot_topics:
                raise Exception("获取热搜数据失败")

            print(f"✅ 成功获取 {len(hot_topics)} 个热搜话题")

            # 3. 分析每个话题并生成创意
            all_ideas = []
            analysis_results = {'hot_topics': hot_topics, 'ideas_with_scores': []}

            print("🔍 开始分析话题背景和生成创意...")
            for i, topic in enumerate(hot_topics, 1):
                print(f"📝 正在分析第 {i}/{len(hot_topics)} 个话题: {topic['title']}")

                try:
                    # 分析话题背景
                    background_info = await analyze_topic_background(
                        topic['title'],
                        web_search_func
                    )

                    # 生成产品创意
                    ideas = await generate_product_ideas(
                        topic,
                        background_info,
                        self.num_ideas_per_topic
                    )

                    # 为每个创意计算评分
                    for idea in ideas:
                        score_details = calculate_product_idea_score(
                            idea,
                            topic,
                            background_info
                        )

                        # 合并创意信息和评分
                        idea_with_score = {**idea, **score_details}
                        all_ideas.append(idea_with_score)

                    print(f"   💡 为话题 '{topic['title']}' 生成了 {len(ideas)} 个创意")

                except Exception as e:
                    print(f"   ❌ 分析话题 '{topic['title']}' 时出错: {e}")
                    continue

            # 4. 按评分排序创意
            all_ideas.sort(key=lambda x: x.get('final_score', 0), reverse=True)
            analysis_results['ideas_with_scores'] = all_ideas

            # 5. 生成统计信息
            stats = self._generate_statistics(analysis_results)
            analysis_results.update(stats)

            print(f"🎉 分析完成！共生成 {len(all_ideas)} 个产品创意")

            return analysis_results

        except Exception as e:
            print(f"❌ 分析过程中发生错误: {e}")
            raise

    def _generate_statistics(self, analysis_results: Dict) -> Dict:
        """
        生成统计信息

        Args:
            analysis_results: 分析结果

        Returns:
            统计信息
        """
        ideas = analysis_results['ideas_with_scores']

        excellent_count = sum(1 for idea in ideas if idea.get('score_category') == 'excellent')
        good_count = sum(1 for idea in ideas if idea.get('score_category') == 'good')
        moderate_count = sum(1 for idea in ideas if idea.get('score_category') == 'moderate')

        total_score = sum(idea.get('final_score', 0) for idea in ideas)
        average_score = round(total_score / len(ideas), 1) if ideas else 0

        return {
            'excellent_count': excellent_count,
            'good_count': good_count,
            'moderate_count': moderate_count,
            'average_score': average_score,
            'total_ideas': len(ideas)
        }

    def generate_report(self, analysis_results: Dict, output_dir: str = None) -> str:
        """
        生成HTML报告

        Args:
            analysis_results: 分析结果
            output_dir: 输出目录

        Returns:
            生成的报告文件路径
        """
        try:
            print("📄 正在生成HTML报告...")

            report_path = generate_analysis_report(analysis_results, output_dir)

            print(f"✅ 报告已生成: {report_path}")
            return report_path

        except Exception as e:
            print(f"❌ 生成报告时出错: {e}")
            raise

    def print_summary(self, analysis_results: Dict):
        """
        打印分析摘要

        Args:
            analysis_results: 分析结果
        """
        ideas = analysis_results['ideas_with_scores']
        hot_topics = analysis_results['hot_topics']

        print("\n" + "="*60)
        print("🎯 微博热搜产品创意分析摘要")
        print("="*60)

        print(f"📊 分析统计:")
        print(f"   • 热搜话题数: {len(hot_topics)}")
        print(f"   • 生成创意数: {len(ideas)}")
        print(f"   • 优秀创意: {analysis_results.get('excellent_count', 0)} 个")
        print(f"   • 良好创意: {analysis_results.get('good_count', 0)} 个")
        print(f"   • 平均评分: {analysis_results.get('average_score', 0)} 分")

        print(f"\n🏆 TOP 5 产品创意:")
        top_ideas = ideas[:5]
        for i, idea in enumerate(top_ideas, 1):
            score = idea.get('final_score', 0)
            name = idea.get('product_name', '未命名')
            topic = idea.get('topic_title', '')
            category = idea.get('score_category', 'unknown')

            category_icons = {
                'excellent': '🌟',
                'good': '⭐',
                'moderate': '💡',
                'limited': '📝'
            }
            icon = category_icons.get(category, '📝')

            print(f"   {i}. {icon} {name} ({score}分)")
            print(f"      来源: {topic}")
            print(f"      等级: {category}")

        print("\n" + "="*60)


async def main():
    """
    主函数 - 演示如何使用分析器
    """
    # 创建分析器实例
    analyzer = WeiboHotSearchAnalyzer()

    # 这里需要实现实际的Web搜索函数
    # 作为示例，我们使用一个模拟的搜索函数
    def mock_web_search(query: str):
        """模拟Web搜索函数"""
        # 在实际使用中，这里应该调用真实的Web Search API
        return [
            {
                'title': f'关于"{query}"的最新报道',
                'url': 'https://example.com/news1',
                'source': '新闻媒体',
                'summary': f'这是一个关于{query}的详细新闻报道...',
                'publish_time': '2024-01-01'
            }
        ]

    try:
        # 执行分析
        analysis_results = await analyzer.run_analysis(
            max_topics=5,  # 示例：只分析前5个话题
            web_search_func=mock_web_search
        )

        # 打印摘要
        analyzer.print_summary(analysis_results)

        # 生成报告
        report_path = analyzer.generate_report(analysis_results)
        print(f"\n📋 详细报告已保存至: {report_path}")

    except Exception as e:
        print(f"❌ 程序执行失败: {e}")
        return 1

    return 0


if __name__ == "__main__":
    # 运行主程序
    exit_code = asyncio.run(main())
    sys.exit(exit_code)


# Claude Code Skill 接口函数
def analyze_weibo_hotsearch(max_topics: int = 10, web_search_func=None):
    """
    Claude Code Skill 接口函数

    Args:
        max_topics: 最大分析话题数量
        web_search_func: Web搜索函数

    Returns:
        分析结果和报告路径
    """
    async def _analyze():
        analyzer = WeiboHotSearchAnalyzer()

        # 执行分析
        analysis_results = await analyzer.run_analysis(max_topics, web_search_func)

        # 生成报告
        report_path = analyzer.generate_report(analysis_results)

        # 打印摘要
        analyzer.print_summary(analysis_results)

        return {
            'analysis_results': analysis_results,
            'report_path': report_path
        }

    # 运行异步分析
    return asyncio.run(_analyze())


def create_argument_parser():
    """创建命令行参数解析器"""
    parser = argparse.ArgumentParser(
        description='微博热搜产品创意分析工具',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
示例用法:
  python main.py                           # 使用默认配置
  python main.py --topics 15               # 分析15个话题
  python main.py --focus tech,lifestyle    # 专注科技和生活领域
  python main.py --weights 70,30          # 调整评分权重
  python main.py --output ./reports       # 指定输出目录
  python main.py --quiet                   # 静默模式

更多信息请访问: https://github.com/your-repo/weibo-hotsearch-idea-analyzer
        """
    )

    # 基础参数
    parser.add_argument(
        '--topics', '-t',
        type=int,
        default=10,
        help='分析的热搜话题数量 (5-20，默认: 10)'
    )

    parser.add_argument(
        '--output', '-o',
        type=str,
        default='./output/reports',
        help='HTML报告输出目录 (默认: ./output/reports)'
    )

    parser.add_argument(
        '--focus', '-f',
        type=str,
        default='all',
        help='专注行业领域，逗号分隔 (tech,lifestyle,finance等，默认: all)'
    )

    parser.add_argument(
        '--weights', '-w',
        type=str,
        default='80,20',
        help='有趣度/有用度权重比，逗号分隔的两个整数 (默认: 80,20)'
    )

    parser.add_argument(
        '--ideas', '-i',
        type=int,
        default=3,
        help='每个话题生成的创意数 (1-5，默认: 3)'
    )

    parser.add_argument(
        '--quiet', '-q',
        action='store_true',
        help='静默模式，减少输出'
    )

    parser.add_argument(
        '--help-extended',
        action='store_true',
        help='显示扩展帮助信息'
    )

    parser.add_argument(
        '--config-file',
        type=str,
        help='从指定文件加载配置'
    )

    parser.add_argument(
        '--save-config',
        type=str,
        help='保存当前配置到指定文件'
    )

    parser.add_argument(
        '--version',
        action='version',
        version='微博热搜产品创意分析工具 v1.0.0'
    )

    return parser


def print_extended_help():
    """打印扩展帮助信息"""
    help_text = """
🔥 微博热搜产品创意分析工具 - 扩展帮助

## 功能特点

### 核心功能
- 📡 实时获取微博热搜数据 (天行数据API)
- 🔍 AI驱动的热点话题背景分析
- 💡 智能产品创意生成
- 📊 专业评分系统 (有趣度80% + 有用度20%)
- 📄 美观的HTML分析报告

### 评分体系详解

#### 有趣度评分 (80%权重)
- 新颖性因素 (25%): 评估产品概念的创新性和独特性
- 参与潜力 (25%): 分析用户互动和社交传播可能性
- 趋势契合度 (15%): 评估与当前热点趋势的匹配度
- 故事潜力 (15%): 分析产品背后的故事性和传播价值

#### 有用度评分 (20%权重)
- 问题解决能力 (10%): 评估解决实际问题的有效性
- 市场可行性 (5%): 分析商业模式和盈利潜力
- 用户价值 (5%): 评估为目标用户创造的价值

### 评分等级
- 🌟 优秀 (80-100分): 强烈推荐开发，市场前景广阔
- ⭐ 良好 (60-79分): 值得考虑，建议进一步调研
- 💡 一般 (40-59分): 有待改进，需要优化核心价值
- 📝 有限 (0-39分): 需要重新评估或寻找新方向

## 行业领域说明

支持以下专注领域:
- tech: 科技、互联网、人工智能
- lifestyle: 生活方式、消费、娱乐
- finance: 金融、投资、理财
- entertainment: 娱乐、影视、游戏
- education: 教育、培训、学习
- all: 全部领域 (默认)

## 输出文件

分析完成后会生成以下文件:
1. **HTML报告**: weibo_analysis_YYYYMMDD_HHMMSS.html
   - 完整的产品创意分析报告
   - 包含统计概览、热点时间线、创意详情
   - 支持浏览器查看，响应式设计

2. **JSON数据**: latest_analysis_result.json
   - 结构化的分析数据
   - 便于程序处理和二次开发

## 环境变量支持

可通过以下环境变量配置默认参数:
- WEIBO_TOPICS: 默认分析话题数量
- WEIBO_OUTPUT_DIR: 默认输出目录
- WEIBO_FOCUS: 默认专注领域
- WEIBO_WEIGHTS: 默认评分权重
- WEIBO_IDEAS: 默认每话题创意数
- WEIBO_QUIET: 默认静默模式

## 故障排除

### 常见问题

1. **API连接失败**
   - 检查网络连接
   - 验证API密钥配置
   - 确认天行数据服务状态

2. **参数错误**
   - 使用 --help 查看参数格式
   - 确认参数值在有效范围内
   - 检查权重参数是否为两个整数且和为100

3. **输出目录错误**
   - 确认目录权限
   - 检查磁盘空间
   - 使用绝对路径或确认相对路径正确

4. **模块导入失败**
   - 确认Python环境完整
   - 检查依赖包是否安装
   - 验证文件结构完整性

### 获取帮助

如果遇到问题，可以:
- 使用 --help 查看基础帮助
- 使用 --help-extended 查看扩展帮助
- 查看日志文件了解详细错误信息
- 提交Issue到项目仓库

## 技术架构

项目采用模块化设计:
- src/core/: 核心功能模块
- src/config/: 配置管理
- src/utils/: 工具函数
- src/templates/: HTML报告模板
- .claude/commands/: 斜杠命令定义

## 开发指南

### 贡献代码
1. Fork 项目
2. 创建功能分支
3. 提交更改
4. 发起 Pull Request

### 扩展功能
- 在 src/core/ 中添加新的分析模块
- 在 src/config/ 中添加新的配置选项
- 更新评分算法在 src/core/scorer.py
- 自定义HTML模板在 src/templates/

## 许可证

本项目采用 MIT 许可证，详见 LICENSE 文件。
    """
    print(help_text)


async def cli_main():
    """命令行主函数"""
    parser = create_argument_parser()
    args = parser.parse_args()

    # 显示扩展帮助
    if args.help_extended:
        print_extended_help()
        return 0

    try:
        # 加载配置文件 (如果指定)
        if args.config_file:
            from command_config import load_config_from_file
            file_config = load_config_from_file(args.config_file)
            if file_config:
                config = file_config
            else:
                print(f"警告: 无法加载配置文件 {args.config_file}，使用默认配置")
                config = CommandConfig()
        else:
            config = CommandConfig()

        # 解析命令行参数
        cli_args = {
            'topics': args.topics,
            'output': args.output,
            'focus': args.focus,
            'weights': args.weights,
            'ideas': args.ideas,
            'quiet': args.quiet
        }

        # 应用命令行参数覆盖
        cli_config = ConfigParser.parse_command_args(cli_args)

        # 合并配置
        final_config = ConfigMerger.merge_configs(config, cli_config.to_dict())

        # 应用环境变量覆盖
        final_config = ConfigMerger.apply_environment_overrides(final_config)

        # 验证输出目录
        from command_config import ConfigValidator
        final_config.output_dir = ConfigValidator.validate_output_directory(final_config.output_dir)

        # 保存配置 (如果指定)
        if args.save_config:
            from command_config import save_config_to_file
            success = save_config_to_file(final_config, args.save_config)
            if success:
                print(f"✅ 配置已保存到: {args.save_config}")
            else:
                print(f"⚠️  配置保存失败")

        # 显示配置信息
        if not final_config.quiet_mode:
            print("🔥 微博热搜产品创意分析工具")
            print("=" * 50)
            print(f"📊 分析配置:")
            print(f"   • 话题数量: {final_config.max_topics}")
            print(f"   • 输出目录: {final_config.output_dir}")
            print(f"   • 专注领域: {final_config.focus_string}")
            print(f"   • 评分权重: {final_config.weights_string}")
            print(f"   • 创意数量: {final_config.ideas_per_topic}")
            print("=" * 50)

        # 创建分析器实例
        analyzer = WeiboHotSearchAnalyzer(final_config)

        # 创建模拟Web搜索函数 (实际使用时应集成真实的搜索API)
        def mock_web_search(query: str):
            """模拟Web搜索函数"""
            return [
                {
                    'title': f'关于"{query}"的最新报道',
                    'url': 'https://example.com/news1',
                    'source': '新闻媒体',
                    'summary': f'这是一个关于{query}的详细新闻报道...',
                    'publish_time': '2024-01-01'
                }
            ]

        # 执行分析
        if not final_config.quiet_mode:
            print("📡 正在获取微博热搜数据...")

        analysis_results = await analyzer.run_analysis(
            max_topics=final_config.max_topics,
            web_search_func=mock_web_search,
            config=final_config
        )

        if not analysis_results or not analysis_results.get('hot_topics'):
            print("❌ 未能获取微博热搜数据")
            return 1

        # 生成报告
        if not final_config.quiet_mode:
            print("📄 正在生成分析报告...")

        report_path = analyzer.generate_report(
            analysis_results,
            output_dir=final_config.output_dir
        )

        # 显示结果
        if not final_config.quiet_mode:
            analyzer.print_summary(analysis_results)
            print(f"\n📋 详细报告已保存至: {report_path}")

        # 保存分析结果
        result_data = {
            'config': final_config.to_dict(),
            'analysis_results': analysis_results,
            'report_path': report_path
        }

        result_file = os.path.join(final_config.output_dir, 'cli_analysis_result.json')
        try:
            import json
            with open(result_file, 'w', encoding='utf-8') as f:
                json.dump(result_data, f, ensure_ascii=False, indent=2, default=str)
        except Exception as e:
            if not final_config.quiet_mode:
                print(f"⚠️  结果保存失败: {e}")

        return 0

    except KeyboardInterrupt:
        print("\n👋 用户中断操作")
        return 130
    except Exception as e:
        print(f"❌ 程序执行失败: {e}")
        if args.help_extended:
            print("\n💡 使用 --help-extended 查看详细帮助信息")
        return 1


if __name__ == "__main__":
    # 命令行模式
    exit_code = asyncio.run(cli_main())
    sys.exit(exit_code)