"""
天行数据微博热搜API集成模块
"""
import requests
import time
from typing import List, Dict, Optional
from src.config.api_config import API_CONFIG, ANALYSIS_CONFIG


class TianAPIClient:
    """天行数据API客户端"""

    def __init__(self):
        self.config = API_CONFIG['tianapi']
        self.base_url = self.config['base_url']
        self.api_key = self.config['api_key']
        self.timeout = self.config['timeout']
        self.retry_attempts = self.config['retry_attempts']

    def fetch_weibo_hotsearch(self, max_topics: int = None) -> Optional[List[Dict]]:
        """
        获取微博热搜榜单

        Args:
            max_topics: 最大返回话题数量，None表示返回所有

        Returns:
            热搜话题列表，失败时返回None
        """
        if max_topics is None:
            max_topics = ANALYSIS_CONFIG['max_topics_to_analyze']

        url = f"{self.base_url}{self.config['weibo_hotsearch_endpoint']}"
        params = {'key': self.api_key}

        for attempt in range(self.retry_attempts):
            try:
                response = requests.get(url, params=params, timeout=self.timeout)
                response.raise_for_status()

                data = response.json()

                if data.get('code') == 200:
                    hot_topics = data.get('newslist', [])
                    # 限制返回话题数量
                    return hot_topics[:max_topics]
                else:
                    print(f"API返回错误: {data.get('msg', '未知错误')}")
                    return None

            except requests.exceptions.RequestException as e:
                print(f"API请求失败 (尝试 {attempt + 1}/{self.retry_attempts}): {e}")
                if attempt < self.retry_attempts - 1:
                    time.sleep(2 ** attempt)  # 指数退避
                else:
                    print("所有重试都失败了")
                    return None
            except Exception as e:
                print(f"处理API响应时发生错误: {e}")
                return None

    def validate_api_response(self, response_data: Dict) -> bool:
        """
        验证API响应数据格式

        Args:
            response_data: API响应数据

        Returns:
            验证结果
        """
        if not isinstance(response_data, dict):
            return False

        # 检查必要字段
        required_fields = ['code', 'newslist']
        for field in required_fields:
            if field not in response_data:
                return False

        # 检查新闻列表格式
        newslist = response_data.get('newslist', [])
        if not isinstance(newslist, list):
            return False

        # 检查每个新闻项的基本格式
        for item in newslist:
            if not isinstance(item, dict):
                return False
            if 'title' not in item:
                return False

        return True

    def format_hot_topics(self, raw_topics: List[Dict]) -> List[Dict]:
        """
        格式化热搜话题数据

        Args:
            raw_topics: 原始热搜数据

        Returns:
            格式化后的热搜数据
        """
        formatted_topics = []

        for i, topic in enumerate(raw_topics, 1):
            formatted_topic = {
                'rank': i,
                'title': topic.get('title', '').strip(),
                'hot_score': topic.get('hot', '0'),
                'raw_data': topic  # 保留原始数据以备后用
            }
            formatted_topics.append(formatted_topic)

        return formatted_topics


# 全局API客户端实例
api_client = TianAPIClient()


def get_weibo_hotsearch(max_topics: int = None) -> Optional[List[Dict]]:
    """
    获取微博热搜榜单（便捷函数）

    Args:
        max_topics: 最大返回话题数量

    Returns:
        格式化后的热搜话题列表
    """
    raw_data = api_client.fetch_weibo_hotsearch(max_topics)
    if raw_data:
        return api_client.format_hot_topics(raw_data)
    return None


def test_api_connection() -> bool:
    """
    测试API连接

    Returns:
            连接是否成功
    """
    try:
        topics = api_client.fetch_weibo_hotsearch(1)  # 只获取一个话题进行测试
        return topics is not None and len(topics) > 0
    except Exception as e:
        print(f"API连接测试失败: {e}")
        return False