"""
AI产品创意生成模块
基于热搜话题背景信息生成创新产品创意
"""
from typing import List, Dict, Optional
from datetime import datetime
import re


class ProductIdeaGenerator:
    """产品创意生成器"""

    def __init__(self):
        self.product_categories = [
            '移动应用', '社交媒体', '电商购物', '教育培训', '娱乐游戏',
            '健康医疗', '金融服务', '企业服务', '智能家居', '出行交通',
            '内容创作', '生活服务', '环保科技', '人工智能', '区块链'
        ]

        self.target_user_groups = [
            '学生群体', '职场人士', '创业者', '家庭用户', '老年人',
            'Z世代', '女性用户', '男性用户', '企业客户', '开发者'
        ]

    async def generate_product_ideas(self, topic_info: Dict, background_info: Dict, num_ideas: int = 3) -> List[Dict]:
        """
        基于话题信息生成产品创意

        Args:
            topic_info: 热搜话题信息
            background_info: 背景信息
            num_ideas: 生成创意数量

        Returns:
            产品创意列表
        """
        ideas = []
        topic_title = topic_info.get('title', '')
        topic_rank = topic_info.get('rank', 0)

        # 分析话题特征
        topic_features = self._analyze_topic_features(topic_title, background_info)

        # 生成多个方向的创意
        for i in range(num_ideas):
            idea = await self._generate_single_idea(topic_info, background_info, topic_features, i + 1)
            if idea:
                ideas.append(idea)

        return ideas

    def _analyze_topic_features(self, topic_title: str, background_info: Dict) -> Dict:
        """
        分析话题特征

        Args:
            topic_title: 话题标题
            background_info: 背景信息

        Returns:
            话题特征分析
        """
        features = {
            'keywords': self._extract_keywords(topic_title),
            'sentiment': background_info.get('sentiment_analysis', {}).get('sentiment_score', 0.5),
            'market_activity': background_info.get('market_context', {}).get('market_activity_level', 'medium'),
            'event_types': list(background_info.get('market_context', {}).get('event_type_distribution', {}).keys()),
            'topic_length': len(topic_title),
            'has_numbers': bool(re.search(r'\d', topic_title)),
            'is_technical': self._is_technical_topic(topic_title, background_info)
        }

        return features

    def _extract_keywords(self, topic_title: str) -> List[str]:
        """
        提取话题关键词

        Args:
            topic_title: 话题标题

        Returns:
            关键词列表
        """
        # 简单的关键词提取
        # 可以集成更复杂的NLP分析
        stop_words = ['的', '了', '在', '是', '我', '有', '和', '就', '不', '人', '都', '一', '一个', '上', '也', '很', '到', '说', '要', '去', '你', '会', '着', '没有', '看', '好', '自己', '这']

        words = re.findall(r'[\u4e00-\u9fff]+', topic_title)
        keywords = [word for word in words if word not in stop_words and len(word) > 1]

        return keywords[:5]  # 返回最多5个关键词

    def _is_technical_topic(self, topic_title: str, background_info: Dict) -> bool:
        """
        判断是否为技术类话题

        Args:
            topic_title: 话题标题
            background_info: 背景信息

        Returns:
            是否为技术类话题
        """
        tech_keywords = ['AI', '人工智能', '机器学习', '大数据', '区块链', '5G', '物联网', '云计算', '自动驾驶', '芯片', '算法', '编程']

        # 检查标题中是否包含技术关键词
        title_has_tech = any(keyword in topic_title for keyword in tech_keywords)

        # 检查事件类型中是否包含技术类事件
        event_types = background_info.get('market_context', {}).get('event_type_distribution', {})
        has_tech_events = event_types.get('technology', 0) > 0

        return title_has_tech or has_tech_events

    async def _generate_single_idea(self, topic_info: Dict, background_info: Dict, features: Dict, idea_index: int) -> Optional[Dict]:
        """
        生成单个产品创意

        Args:
            topic_info: 话题信息
            background_info: 背景信息
            features: 话题特征
            idea_index: 创意索引

        Returns:
            产品创意
        """
        try:
            topic_title = topic_info.get('title', '')

            # 选择创意方向
            creative_direction = self._select_creative_direction(features, idea_index)

            # 生成产品名称
            product_name = self._generate_product_name(topic_title, creative_direction, idea_index)

            # 定义核心功能
            core_functions = self._define_core_functions(topic_title, features, creative_direction)

            # 定义目标用户
            target_users = self._define_target_users(features, background_info)

            # 生成产品描述
            product_description = self._generate_product_description(topic_title, core_functions, target_users)

            idea = {
                'idea_id': f"idea_{topic_info.get('rank', 0)}_{idea_index}",
                'topic_title': topic_title,
                'topic_rank': topic_info.get('rank', 0),
                'product_name': product_name,
                'creative_direction': creative_direction,
                'core_functions': core_functions,
                'target_users': target_users,
                'product_description': product_description,
                'generated_time': datetime.now().isoformat(),
                'idea_score': None  # 将在评分模块中填写
            }

            return idea

        except Exception as e:
            print(f"生成创意时发生错误: {e}")
            return None

    def _select_creative_direction(self, features: Dict, idea_index: int) -> str:
        """
        选择创意方向

        Args:
            features: 话题特征
            idea_index: 创意索引

        Returns:
            创意方向
        """
        directions = [
            '问题解决方案', '娱乐社交', '效率工具', '学习成长', '商业模式创新'
        ]

        # 基于话题特征选择方向
        if features['sentiment'] > 0.7:
            # 正面话题倾向于娱乐和社交
            if idea_index == 1:
                return '娱乐社交'
            else:
                return '商业模式创新'
        elif features['sentiment'] < 0.3:
            # 负面话题倾向于解决方案
            return '问题解决方案'
        elif features['is_technical']:
            # 技术话题倾向于效率工具
            return '效率工具'
        else:
            # 其他情况轮流选择不同方向
            return directions[(idea_index - 1) % len(directions)]

    def _generate_product_name(self, topic_title: str, direction: str, idea_index: int) -> str:
        """
        生成产品名称

        Args:
            topic_title: 话题标题
            direction: 创意方向
            idea_index: 创意索引

        Returns:
            产品名称
        """
        keywords = self._extract_keywords(topic_title)
        if not keywords:
            keywords = ['智能']

        # 名称模板
        name_templates = {
            '问题解决方案': ['{keyword}助手', '{keyword}解决方案', '智解{keyword}'],
            '娱乐社交': ['{keyword}圈', '{keyword}社区', '{keyword}空间'],
            '效率工具': ['{keyword}工具', '快{keyword}', '{keyword}管家'],
            '学习成长': ['{keyword}学院', '学{keyword}', '{keyword}课堂'],
            '商业模式创新': ['{keyword}平台', '{keyword}+', '新{keyword}']
        }

        templates = name_templates.get(direction, ['{keyword}App', '智能{keyword}'])
        template = templates[(idea_index - 1) % len(templates)]
        keyword = keywords[0] if keywords else '智能'

        product_name = template.format(keyword=keyword)

        # 确保名称长度合理
        if len(product_name) > 10:
            product_name = product_name[:10]

        return product_name

    def _define_core_functions(self, topic_title: str, features: Dict, direction: str) -> List[str]:
        """
        定义核心功能

        Args:
            topic_title: 话题标题
            features: 话题特征
            direction: 创意方向

        Returns:
            核心功能列表
        """
        # 基于方向的基础功能
        base_functions = {
            '问题解决方案': [
                f'智能分析{topic_title}相关问题',
                '提供个性化解决方案',
                '解决方案效果追踪'
            ],
            '娱乐社交': [
                f'{topic_title}内容分享',
                '兴趣匹配推荐',
                '社区互动交流'
            ],
            '效率工具': [
                f'{topic_title}信息整合',
                '自动化处理流程',
                '数据可视化分析'
            ],
            '学习成长': [
                f'{topic_title}知识课程',
                '互动学习体验',
                '学习进度追踪'
            ],
            '商业模式创新': [
                f'{topic_title}资源对接',
                '价值创造平台',
                '生态系统构建'
            ]
        }

        functions = base_functions.get(direction, [
            f'{topic_title}信息管理',
            '智能推荐系统',
            '用户互动功能'
        ])

        # 根据话题特征调整功能
        if features['has_numbers']:
            functions.append('数据统计分析')

        if features['is_technical']:
            functions.append('技术实现方案')

        return functions[:5]  # 返回最多5个核心功能

    def _define_target_users(self, features: Dict, background_info: Dict) -> Dict:
        """
        定义目标用户

        Args:
            features: 话题特征
            background_info: 背景信息

        Returns:
            目标用户信息
        """
        # 基于话题特征选择用户群体
        if features['is_technical']:
            primary_users = ['开发者', '技术爱好者', '企业客户']
        elif features['sentiment'] > 0.6:
            primary_users = ['年轻人', '社交媒体用户', '内容创作者']
        else:
            primary_users = ['职场人士', '学生群体', '普通用户']

        # 检查事件类型来细化用户群体
        event_types = features.get('event_types', [])
        if 'business' in event_types:
            primary_users = ['创业者', '企业管理者', '投资者']
        elif 'social' in event_types:
            primary_users = ['家庭用户', '教育工作者', '公益组织']

        user_persona = self._generate_user_persona(primary_users[0], features)

        return {
            'primary_users': primary_users,
            'user_persona': user_persona,
            'estimated_user_count': self._estimate_user_count(features, background_info)
        }

    def _generate_user_persona(self, user_type: str, features: Dict) -> str:
        """
        生成用户画像

        Args:
            user_type: 用户类型
            features: 话题特征

        Returns:
            用户画像描述
        """
        personas = {
            '开发者': '关注技术创新和效率提升，喜欢尝试新工具，重视产品功能性和扩展性',
            '年轻人': '追求时尚和个性化，活跃于社交媒体，重视用户体验和社交互动',
            '职场人士': '注重效率和职业发展，时间宝贵，需要简洁实用的解决方案',
            '创业者': '寻找商业机会，关注市场趋势和投资回报，重视创新和竞争优势'
        }

        return personas.get(user_type, '对新鲜事物感兴趣，追求便捷和高效的数字体验')

    def _estimate_user_count(self, features: Dict, background_info: Dict) -> str:
        """
        估算用户数量

        Args:
            features: 话题特征
            background_info: 背景信息

        Returns:
            用户数量估算
        """
        market_activity = background_info.get('market_context', {}).get('market_activity_level', 'medium')
        news_sources = len(background_info.get('news_sources', []))

        # 基于市场活跃度和新闻数量估算潜在用户
        if market_activity == 'high' and news_sources > 10:
            return '100万+'
        elif market_activity == 'medium' and news_sources > 5:
            return '50万-100万'
        else:
            return '10万-50万'

    def _generate_product_description(self, topic_title: str, core_functions: List[str], target_users: Dict) -> str:
        """
        生成产品描述

        Args:
            topic_title: 话题标题
            core_functions: 核心功能
            target_users: 目标用户

        Returns:
            产品描述
        """
        primary_users = ', '.join(target_users['primary_users'][:2])

        description = f"""
        针对"{topic_title}"热点话题，我们打造了一款创新产品，主要服务于{primary_users}等用户群体。

        核心功能包括：
        {chr(10).join([f"• {func}" for func in core_functions[:3]])}

        产品致力于解决当前市场痛点，通过智能化的方式提升用户体验，创造独特价值。
        """

        return description.strip()


# 全局创意生成器实例
idea_generator = ProductIdeaGenerator()


async def generate_product_ideas(topic_info: Dict, background_info: Dict, num_ideas: int = 3) -> List[Dict]:
    """
    生成产品创意（便捷函数）

    Args:
        topic_info: 话题信息
        background_info: 背景信息
        num_ideas: 创意数量

    Returns:
        产品创意列表
    """
    return await idea_generator.generate_product_ideas(topic_info, background_info, num_ideas)