"""
产品创意评分算法模块
实现有趣度80% + 有用度20%的评分体系
"""
from typing import Dict, List
from src.config.api_config import SCORING_WEIGHTS


class ProductIdeaScorer:
    """产品创意评分器"""

    def __init__(self):
        self.interestingness_weight = SCORING_WEIGHTS['interestingness']
        self.usefulness_weight = SCORING_WEIGHTS['usefulness']

    def calculate_idea_score(self, idea: Dict, topic_info: Dict, background_info: Dict) -> Dict:
        """
        计算产品创意的综合评分

        Args:
            idea: 产品创意信息
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            包含详细评分信息的字典
        """
        # 计算有趣度评分 (80%权重)
        interestingness_score = self._calculate_interestingness_score(idea, topic_info, background_info)

        # 计算有用度评分 (20%权重)
        usefulness_score = self._calculate_usefulness_score(idea, topic_info, background_info)

        # 计算最终评分
        final_score = self._calculate_final_score(interestingness_score, usefulness_score)

        # 评分等级
        score_category = self._categorize_score(final_score)

        # 详细评分分析
        score_details = {
            'final_score': round(final_score, 2),
            'score_category': score_category,
            'interestingness_score': round(interestingness_score, 2),
            'usefulness_score': round(usefulness_score, 2),
            'interestingness_breakdown': self._get_interestingness_breakdown(idea, topic_info, background_info),
            'usefulness_breakdown': self._get_usefulness_breakdown(idea, topic_info, background_info),
            'scoring_time': self._get_current_time(),
            'recommendation': self._generate_recommendation(final_score, score_category)
        }

        return score_details

    def _calculate_interestingness_score(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        计算有趣度评分 (80%权重)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            有趣度评分 (0-100)
        """
        # 新颖性因素 (25%)
        novelty_score = self._assess_novelty(idea, topic_info, background_info)

        # 参与潜力 (25%)
        engagement_score = self._assess_engagement_potential(idea, topic_info, background_info)

        # 趋势契合度 (15%)
        trend_alignment_score = self._assess_trend_alignment(idea, topic_info, background_info)

        # 故事潜力 (15%)
        storytelling_score = self._assess_storytelling_potential(idea, topic_info, background_info)

        # 加权计算
        interestingness_score = (
            novelty_score * 0.3125 +      # 25% / 80%
            engagement_score * 0.3125 +   # 25% / 80%
            trend_alignment_score * 0.1875 +  # 15% / 80%
            storytelling_score * 0.1875       # 15% / 80%
        )

        return interestingness_score

    def _calculate_usefulness_score(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        计算有用度评分 (20%权重)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            有用度评分 (0-100)
        """
        # 问题解决能力 (10%)
        problem_solving_score = self._assess_problem_solving(idea, topic_info, background_info)

        # 市场可行性 (5%)
        market_viability_score = self._assess_market_viability(idea, topic_info, background_info)

        # 用户价值 (5%)
        user_value_score = self._assess_user_value(idea, topic_info, background_info)

        # 加权计算
        usefulness_score = (
            problem_solving_score * 0.5 +        # 10% / 20%
            market_viability_score * 0.25 +      # 5% / 20%
            user_value_score * 0.25              # 5% / 20%
        )

        return usefulness_score

    def _assess_novelty(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        评估新颖性 (25%)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            新颖性评分 (0-100)
        """
        score = 50  # 基础分

        product_name = idea.get('product_name', '').lower()
        creative_direction = idea.get('creative_direction', '')
        core_functions = idea.get('core_functions', [])

        # 创意方向新颖性
        if creative_direction == '商业模式创新':
            score += 20
        elif creative_direction == '问题解决方案':
            score += 15

        # 产品名称独特性
        if any(keyword in product_name for keyword in ['智能', 'AI', '+', '新']):
            score += 10

        # 功能创新性
        if len(core_functions) >= 3:
            score += 10
        if len(core_functions) >= 5:
            score += 5

        # 话题热度影响
        topic_rank = topic_info.get('rank', 50)
        if topic_rank <= 5:
            score += 10
        elif topic_rank <= 10:
            score += 5

        return min(100, score)

    def _assess_engagement_potential(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        评估参与潜力 (25%)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            参与潜力评分 (0-100)
        """
        score = 50  # 基础分

        creative_direction = idea.get('creative_direction', '')
        target_users = idea.get('target_users', {})
        core_functions = idea.get('core_functions', [])

        # 创意方向参与度
        if creative_direction in ['娱乐社交', '内容创作']:
            score += 25
        elif creative_direction in ['学习成长', '效率工具']:
            score += 15

        # 用户群体广泛性
        primary_users = target_users.get('primary_users', [])
        if '年轻人' in primary_users or '普通用户' in primary_users:
            score += 15
        elif len(primary_users) >= 3:
            score += 10

        # 社交功能
        has_social_features = any('社交' in func or '分享' in func or '互动' in func for func in core_functions)
        if has_social_features:
            score += 10

        # 话题情感积极度
        sentiment_score = background_info.get('sentiment_analysis', {}).get('sentiment_score', 0.5)
        if sentiment_score > 0.7:
            score += 10
        elif sentiment_score < 0.3:
            score -= 10

        return min(100, max(0, score))

    def _assess_trend_alignment(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        评估趋势契合度 (15%)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            趋势契合度评分 (0-100)
        """
        score = 50  # 基础分

        # 市场活跃度
        market_activity = background_info.get('market_context', {}).get('market_activity_level', 'medium')
        if market_activity == 'high':
            score += 30
        elif market_activity == 'medium':
            score += 15

        # 事件类型分布
        event_types = background_info.get('market_context', {}).get('event_type_distribution', {})
        if 'technology' in event_types and event_types['technology'] > 0:
            score += 10
        if 'business' in event_types and event_types['business'] > 0:
            score += 10

        # 新闻源数量
        news_sources = background_info.get('news_sources', [])
        if len(news_sources) > 10:
            score += 10
        elif len(news_sources) > 5:
            score += 5

        return min(100, score)

    def _assess_storytelling_potential(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        评估故事潜力 (15%)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            故事潜力评分 (0-100)
        """
        score = 50  # 基础分

        # 话题情感色彩
        sentiment_analysis = background_info.get('sentiment_analysis', {})
        positive_events = sentiment_analysis.get('positive_events', 0)
        negative_events = sentiment_analysis.get('negative_events', 0)

        if positive_events > negative_events:
            score += 20  # 正面故事更有传播力
        elif negative_events > positive_events * 2:
            score += 15  # 问题和解决方案的故事性

        # 关键事件数量
        key_events = background_info.get('key_events', [])
        if len(key_events) > 5:
            score += 15
        elif len(key_events) > 2:
            score += 10

        # 产品描述的情感元素
        product_description = idea.get('product_description', '')
        emotional_keywords = ['创新', '革命', '突破', '改变', '影响', '未来']
        emotional_count = sum(1 for keyword in emotional_keywords if keyword in product_description)
        score += min(15, emotional_count * 3)

        return min(100, score)

    def _assess_problem_solving(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        评估问题解决能力 (10%)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            问题解决能力评分 (0-100)
        """
        score = 50  # 基础分

        creative_direction = idea.get('creative_direction', '')
        core_functions = idea.get('core_functions', [])

        # 创意方向问题导向性
        if creative_direction == '问题解决方案':
            score += 30
        elif creative_direction == '效率工具':
            score += 20

        # 功能解决问题能力
        problem_keywords = ['解决', '优化', '提升', '改善', '自动化', '智能']
        problem_functions = sum(1 for func in core_functions if any(keyword in func for keyword in problem_keywords))
        score += min(30, problem_functions * 10)

        # 话题问题的紧迫性
        sentiment_score = background_info.get('sentiment_analysis', {}).get('sentiment_score', 0.5)
        if sentiment_score < 0.3:  # 负面话题通常更需要解决方案
            score += 20

        return min(100, score)

    def _assess_market_viability(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        评估市场可行性 (5%)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            市场可行性评分 (0-100)
        """
        score = 60  # 基础分

        target_users = idea.get('target_users', {})
        estimated_user_count = target_users.get('estimated_user_count', '')

        # 用户规模估算
        if '100万+' in estimated_user_count:
            score += 25
        elif '50万' in estimated_user_count:
            score += 20
        elif '10万' in estimated_user_count:
            score += 10

        # 商业模式创新可行性
        if idea.get('creative_direction') == '商业模式创新':
            score += 15

        return min(100, score)

    def _assess_user_value(self, idea: Dict, topic_info: Dict, background_info: Dict) -> float:
        """
        评估用户价值 (5%)

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            用户价值评分 (0-100)
        """
        score = 60  # 基础分

        core_functions = idea.get('core_functions', [])
        target_users = idea.get('target_users', {})

        # 功能价值密度
        if len(core_functions) >= 3:
            score += 20
        elif len(core_functions) >= 2:
            score += 10

        # 用户群体明确性
        primary_users = target_users.get('primary_users', [])
        if len(primary_users) >= 2:
            score += 15

        # 个性化元素
        has_personalization = any('个性化' in func or '定制' in func or '推荐' in func for func in core_functions)
        if has_personalization:
            score += 5

        return min(100, score)

    def _calculate_final_score(self, interestingness_score: float, usefulness_score: float) -> float:
        """
        计算最终评分

        Args:
            interestingness_score: 有趣度评分
            usefulness_score: 有用度评分

        Returns:
            最终评分 (0-100)
        """
        final_score = (
            interestingness_score * self.interestingness_weight +
            usefulness_score * self.usefulness_weight
        )
        return round(final_score, 2)

    def _categorize_score(self, score: float) -> str:
        """
        分类评分等级

        Args:
            score: 评分

        Returns:
            评分等级
        """
        if score >= 80:
            return 'excellent'
        elif score >= 60:
            return 'good'
        elif score >= 40:
            return 'moderate'
        else:
            return 'limited'

    def _get_interestingness_breakdown(self, idea: Dict, topic_info: Dict, background_info: Dict) -> Dict:
        """
        获取有趣度评分明细

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            有趣度评分明细
        """
        return {
            'novelty': self._assess_novelty(idea, topic_info, background_info),
            'engagement': self._assess_engagement_potential(idea, topic_info, background_info),
            'trend_alignment': self._assess_trend_alignment(idea, topic_info, background_info),
            'storytelling': self._assess_storytelling_potential(idea, topic_info, background_info)
        }

    def _get_usefulness_breakdown(self, idea: Dict, topic_info: Dict, background_info: Dict) -> Dict:
        """
        获取有用度评分明细

        Args:
            idea: 产品创意
            topic_info: 话题信息
            background_info: 背景信息

        Returns:
            有用度评分明细
        """
        return {
            'problem_solving': self._assess_problem_solving(idea, topic_info, background_info),
            'market_viability': self._assess_market_viability(idea, topic_info, background_info),
            'user_value': self._assess_user_value(idea, topic_info, background_info)
        }

    def _get_current_time(self) -> str:
        """
        获取当前时间

        Returns:
            当前时间字符串
        """
        from datetime import datetime
        return datetime.now().isoformat()

    def _generate_recommendation(self, score: float, category: str) -> str:
        """
        生成推荐意见

        Args:
            score: 评分
            category: 评分等级

        Returns:
            推荐意见
        """
        recommendations = {
            'excellent': '强烈推荐：这是极具潜力的产品创意，建议优先开发，市场前景广阔',
            'good': '值得考虑：产品创意具有较好价值，建议进一步调研和规划',
            'moderate': '有待改进：创意有一定潜力，需要优化和完善核心价值',
            'limited': '需要重新评估：创意价值有限，建议寻找新的方向或大幅改进'
        }

        return recommendations.get(category, '评分分析中，请等待详细评估')


# 全局评分器实例
scorer = ProductIdeaScorer()


def calculate_product_idea_score(idea: Dict, topic_info: Dict, background_info: Dict) -> Dict:
    """
    计算产品创意评分（便捷函数）

    Args:
        idea: 产品创意
        topic_info: 话题信息
        background_info: 背景信息

    Returns:
        评分详情
    """
    return scorer.calculate_idea_score(idea, topic_info, background_info)