"""
HTML报告生成器
将分析结果转换为美观的HTML报告
"""
import os
from typing import List, Dict
from datetime import datetime
from jinja2 import Template


class HTMLReportGenerator:
    """HTML报告生成器"""

    def __init__(self):
        self.template_path = os.path.join(
            os.path.dirname(__file__), '..', 'templates', 'report_template.html'
        )

    def generate_report(self, analysis_results: Dict) -> str:
        """
        生成完整的HTML报告

        Args:
            analysis_results: 分析结果数据

        Returns:
            HTML报告内容
        """
        # 准备模板数据
        template_data = self._prepare_template_data(analysis_results)

        # 读取模板
        template_content = self._read_template()
        template = Template(template_content)

        # 渲染报告
        html_content = template.render(**template_data)

        return html_content

    def _prepare_template_data(self, analysis_results: Dict) -> Dict:
        """
        准备模板数据

        Args:
            analysis_results: 分析结果

        Returns:
            模板数据
        """
        hot_topics = analysis_results.get('hot_topics', [])
        ideas_with_scores = analysis_results.get('ideas_with_scores', [])

        # 统计信息
        excellent_count = sum(1 for idea in ideas_with_scores if idea.get('score_category') == 'excellent')
        good_count = sum(1 for idea in ideas_with_scores if idea.get('score_category') == 'good')
        moderate_count = sum(1 for idea in ideas_with_scores if idea.get('score_category') == 'moderate')

        total_score = sum(idea.get('final_score', 0) for idea in ideas_with_scores)
        average_score = round(total_score / len(ideas_with_scores), 1) if ideas_with_scores else 0

        # 分类创意
        excellent_ideas = [idea for idea in ideas_with_scores if idea.get('score_category') == 'excellent']
        good_ideas = [idea for idea in ideas_with_scores if idea.get('score_category') == 'good']
        moderate_ideas = [idea for idea in ideas_with_scores if idea.get('score_category') in ['moderate', 'limited']]

        # 生成时间线内容
        timeline_content = self._generate_timeline_content(hot_topics)

        # 生成创意卡片内容
        excellent_ideas_content = self._generate_ideas_content(excellent_ideas, 'excellent')
        good_ideas_content = self._generate_ideas_content(good_ideas, 'good')
        moderate_ideas_content = self._generate_ideas_content(moderate_ideas, 'moderate')

        template_data = {
            'timestamp': datetime.now().strftime('%Y年%m月%d日 %H:%M'),
            'topic_count': len(hot_topics),
            'idea_count': len(ideas_with_scores),
            'excellent_count': excellent_count,
            'good_count': good_count,
            'moderate_count': moderate_count,
            'average_score': average_score,
            'timeline_content': timeline_content,
            'excellent_ideas_content': excellent_ideas_content,
            'good_ideas_content': good_ideas_content,
            'moderate_ideas_content': moderate_ideas_content,
            'has_excellent_ideas': len(excellent_ideas) > 0,
            'has_good_ideas': len(good_ideas) > 0,
            'has_moderate_ideas': len(moderate_ideas) > 0,
            'excellent_ideas': excellent_ideas,
            'good_ideas': good_ideas,
            'moderate_ideas': moderate_ideas
        }

        return template_data

    def _read_template(self) -> str:
        """
        读取HTML模板文件

        Returns:
            模板内容
        """
        try:
            with open(self.template_path, 'r', encoding='utf-8') as f:
                return f.read()
        except FileNotFoundError:
            # 如果模板文件不存在，返回基础模板
            return self._get_fallback_template()
        except Exception as e:
            print(f"读取模板文件失败: {e}")
            return self._get_fallback_template()

    def _generate_timeline_content(self, hot_topics: List[Dict]) -> str:
        """
        生成时间线内容

        Args:
            hot_topics: 热搜话题列表

        Returns:
            时间线HTML内容
        """
        timeline_html = ""

        for i, topic in enumerate(hot_topics[:10]):  # 最多显示10个话题
            rank = topic.get('rank', i + 1)
            title = topic.get('title', '')
            hot_score = topic.get('hot_score', '')

            timeline_html += f"""
            <div class="timeline-item">
                <div class="timeline-dot"></div>
                <div class="timeline-content">
                    <h4 style="color: #333; margin-bottom: 8px;">第{rank}名: {title}</h4>
                    <p style="color: #666; font-size: 0.9em;">
                        热度值: {hot_score}
                    </p>
                    <p style="color: #888; font-size: 0.8em; margin-top: 8px;">
                        该话题引发了广泛讨论，产生了多个产品创意机会
                    </p>
                </div>
            </div>
            """

        return timeline_html

    def _generate_ideas_content(self, ideas: List[Dict], category: str) -> str:
        """
        生成创意卡片内容

        Args:
            ideas: 创意列表
            category: 创意分类

        Returns:
            创意卡片HTML内容
        """
        ideas_html = ""

        for idea in ideas:
            idea_html = self._generate_single_idea_card(idea, category)
            ideas_html += idea_html

        return ideas_html

    def _generate_single_idea_card(self, idea: Dict, category: str) -> str:
        """
        生成单个创意卡片

        Args:
            idea: 创意信息
            category: 创意分类

        Returns:
            创意卡片HTML
        """
        product_name = idea.get('product_name', '未命名产品')
        topic_title = idea.get('topic_title', '')
        topic_rank = idea.get('topic_rank', 0)
        final_score = idea.get('final_score', 0)
        creative_direction = idea.get('creative_direction', '')
        core_functions = idea.get('core_functions', [])
        target_users = idea.get('target_users', {})
        product_description = idea.get('product_description', '')
        recommendation = idea.get('recommendation', '')

        # 目标用户信息
        primary_users = target_users.get('primary_users', [])
        estimated_user_count = target_users.get('estimated_user_count', '')

        # 生成功能列表HTML
        functions_html = ""
        for func in core_functions[:5]:  # 最多显示5个功能
            functions_html += f"<li>{func}</li>"

        # 生成用户群体HTML
        users_html = ""
        for user in primary_users[:3]:  # 最多显示3个用户群体
            users_html += f"<span class='meta-item'>{user}</span>"

        recommendation_style = self._get_recommendation_style(category)

        idea_card_html = f"""
        <div class="idea-card {category}">
            <div class="idea-header">
                <div class="idea-title">{product_name}</div>
                <div class="idea-score">{final_score}</div>
            </div>

            <div class="idea-meta">
                <span class="meta-item">🔥 热搜第{topic_rank}名</span>
                <span class="meta-item">💡 {creative_direction}</span>
                <span class="meta-item">👥 {estimated_user_count}用户</span>
            </div>

            <div class="idea-section">
                <h4>🎯 核心功能</h4>
                <ul>
                    {functions_html}
                </ul>
            </div>

            <div class="idea-section">
                <h4>👤 目标用户</h4>
                <div style="display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 10px;">
                    {users_html}
                </div>
            </div>

            <div class="idea-section">
                <h4>📝 产品描述</h4>
                <div class="idea-description">{product_description}</div>
            </div>

            <div class="idea-section" style="{recommendation_style}">
                <h4>💭 分析建议</h4>
                <p style="color: #666; font-style: italic; line-height: 1.5;">{recommendation}</p>
            </div>
        </div>
        """

        return idea_card_html

    def _get_recommendation_style(self, category: str) -> str:
        """
        获取推荐意见的样式

        Args:
            category: 创意分类

        Returns:
            样式字符串
        """
        styles = {
            'excellent': 'background: linear-gradient(135deg, #FFD70020, #FFA50020); padding: 15px; border-radius: 8px; border-left: 4px solid #FFD700;',
            'good': 'background: linear-gradient(135deg, #4A90E220, #357ABD20); padding: 15px; border-radius: 8px; border-left: 4px solid #4A90E2;',
            'moderate': 'background: linear-gradient(135deg, #95A5A620, #7F8C8D20); padding: 15px; border-radius: 8px; border-left: 4px solid #95A5A6;'
        }
        return styles.get(category, '')

    def _get_fallback_template(self) -> str:
        """
        获取备用模板

        Returns:
            备用模板HTML
        """
        return """
        <!DOCTYPE html>
        <html lang="zh-CN">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>微博热搜产品创意分析报告</title>
            <style>
                body { font-family: 'Microsoft YaHei', Arial, sans-serif; margin: 40px; background: #f5f5f5; }
                .container { max-width: 1000px; margin: 0 auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 5px 20px rgba(0,0,0,0.1); }
                h1 { color: #333; text-align: center; margin-bottom: 30px; }
                h2 { color: #667eea; border-bottom: 2px solid #667eea; padding-bottom: 10px; margin-top: 30px; }
                .stats { display: flex; justify-content: space-around; margin: 30px 0; }
                .stat { text-align: center; padding: 20px; background: #f8f9fa; border-radius: 8px; }
                .stat-number { font-size: 2em; font-weight: bold; color: #667eea; }
                .idea-card { background: #f8f9fa; margin: 20px 0; padding: 20px; border-radius: 8px; border-left: 4px solid #667eea; }
                .score { font-weight: bold; color: #667eea; }
            </style>
        </head>
        <body>
            <div class="container">
                <h1>微博热搜产品创意分析报告</h1>
                <p>生成时间: {{ timestamp }}</p>
                <p>分析话题数: {{ topic_count }} | 创意总数: {{ idea_count }}</p>

                <div class="stats">
                    <div class="stat">
                        <div class="stat-number">{{ excellent_count }}</div>
                        <div>优秀创意</div>
                    </div>
                    <div class="stat">
                        <div class="stat-number">{{ good_count }}</div>
                        <div>良好创意</div>
                    </div>
                    <div class="stat">
                        <div class="stat-number">{{ average_score }}</div>
                        <div>平均评分</div>
                    </div>
                </div>

                <h2>产品创意详情</h2>
                {% for idea in excellent_ideas + good_ideas + moderate_ideas %}
                <div class="idea-card">
                    <h3>{{ idea.product_name }} <span class="score">({{ idea.final_score }}分)</span></h3>
                    <p><strong>来源:</strong> 热搜第{{ idea.topic_rank }}名 - {{ idea.topic_title }}</p>
                    <p><strong>方向:</strong> {{ idea.creative_direction }}</p>
                    <p><strong>核心功能:</strong></p>
                    <ul>
                        {% for func in idea.core_functions[:5] %}
                        <li>{{ func }}</li>
                        {% endfor %}
                    </ul>
                    <p><strong>产品描述:</strong></p>
                    <p>{{ idea.product_description }}</p>
                    <p><strong>分析建议:</strong> {{ idea.recommendation }}</p>
                </div>
                {% endfor %}
            </div>
        </body>
        </html>
        """

    def save_report(self, html_content: str, output_path: str) -> str:
        """
        保存HTML报告

        Args:
            html_content: HTML内容
            output_path: 输出路径

        Returns:
            保存的文件路径
        """
        try:
            # 确保输出目录存在
            os.makedirs(os.path.dirname(output_path), exist_ok=True)

            # 保存文件
            with open(output_path, 'w', encoding='utf-8') as f:
                f.write(html_content)

            return output_path
        except Exception as e:
            print(f"保存报告失败: {e}")
            raise

    def generate_filename(self, prefix: str = "weibo_analysis") -> str:
        """
        生成报告文件名

        Args:
            prefix: 文件名前缀

        Returns:
            文件名
        """
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        return f"{prefix}_{timestamp}.html"


# 全局报告生成器实例
html_generator = HTMLReportGenerator()


def generate_analysis_report(analysis_results: Dict, output_dir: str = None) -> str:
    """
    生成分析报告（便捷函数）

    Args:
        analysis_results: 分析结果数据
        output_dir: 输出目录

    Returns:
        生成的报告文件路径
    """
    # 生成HTML内容
    html_content = html_generator.generate_report(analysis_results)

    # 确定输出路径
    if output_dir is None:
        output_dir = os.path.join(
            os.path.dirname(__file__), '..', '..', 'output', 'reports'
        )

    filename = html_generator.generate_filename()
    output_path = os.path.join(output_dir, filename)

    # 保存报告
    return html_generator.save_report(html_content, output_path)