"""
测试斜杠命令功能
"""
import asyncio
import sys
import os

# 添加当前目录到Python路径
current_dir = os.path.dirname(__file__)
sys.path.insert(0, current_dir)

try:
    from command_config import CommandConfig, ConfigParser, ConfigValidator
    from claude_command_integration import main
except ImportError:
    # 处理相对导入问题
    sys.path.append(current_dir)
    from command_config import CommandConfig, ConfigParser, ConfigValidator
    from claude_command_integration import main


def test_config_parser():
    """测试配置解析"""
    print("🧪 测试配置解析功能")
    print("=" * 50)

    test_cases = [
        # 基础配置
        {
            'name': '基础配置',
            'args': {},
            'expected': {
                'max_topics': 10,
                'ideas_per_topic': 3,
                'quiet_mode': False
            }
        },
        # 自定义话题数量
        {
            'name': '自定义话题数量',
            'args': {'topics': 15},
            'expected': {
                'max_topics': 15,
                'ideas_per_topic': 3
            }
        },
        # 自定义权重
        {
            'name': '自定义权重',
            'args': {'weights': '70,30'},
            'expected': {
                'scoring_weights': (70, 30)
            }
        },
        # 自定义专注领域
        {
            'name': '自定义专注领域',
            'args': {'focus': 'tech,lifestyle'},
            'expected': {
                'industry_focus': ['tech', 'lifestyle']
            }
        }
    ]

    for test_case in test_cases:
        try:
            print(f"📝 测试: {test_case['name']}")
            config = ConfigParser.parse_command_args(test_case['args'])

            # 验证期望值
            for key, expected_value in test_case['expected'].items():
                actual_value = getattr(config, key)
                if actual_value == expected_value:
                    print(f"   ✅ {key}: {actual_value}")
                else:
                    print(f"   ❌ {key}: 期望 {expected_value}, 实际 {actual_value}")

            print()

        except Exception as e:
            print(f"   ❌ 测试失败: {e}")
            print()


def test_config_validation():
    """测试配置验证"""
    print("🧪 测试配置验证功能")
    print("=" * 50)

    error_test_cases = [
        # 无效的话题数量
        {
            'name': '话题数量过小',
            'args': {'topics': 3},
            'expected_error': '话题数量必须在5-20之间'
        },
        # 无效的权重
        {
            'name': '权重和不等于100',
            'args': {'weights': '80,30'},
            'expected_error': '权重之和必须为100'
        },
        # 无效的行业领域
        {
            'name': '无效行业领域',
            'args': {'focus': 'invalid_industry'},
            'expected_error': '无效的行业领域'
        }
    ]

    for test_case in error_test_cases:
        try:
            print(f"📝 测试: {test_case['name']}")
            config = ConfigParser.parse_command_args(test_case['args'])
            errors = config.validate()

            if errors:
                print(f"   ✅ 正确捕获错误: {errors[0]}")
            else:
                print(f"   ❌ 未能捕获预期错误")

        except Exception as e:
            if test_case['expected_error'] in str(e):
                print(f"   ✅ 正确捕获错误: {e}")
            else:
                print(f"   ❌ 错误信息不匹配: {e}")

        print()


def test_command_integration():
    """测试命令集成"""
    print("🧪 测试命令集成功能")
    print("=" * 50)

    test_args = {
        'topics': 5,  # 减少话题数量以加快测试
        'quiet': True  # 静默模式
    }

    try:
        print("📝 测试斜杠命令集成")
        print("   参数:", test_args)

        # 异步执行测试
        async def run_test():
            result = await main(test_args)
            print(f"   📊 执行结果: {'成功' if result.get('success') else '失败'}")
            if result.get('success'):
                print(f"   📋 报告路径: {result.get('report_path')}")
                summary = result.get('summary', {})
                print(f"   📈 统计: {summary}")
            else:
                print(f"   ❌ 错误: {result.get('message')}")
                for suggestion in result.get('suggestions', []):
                    print(f"   💡 建议: {suggestion}")

        asyncio.run(run_test())

    except Exception as e:
        print(f"   ❌ 集成测试失败: {e}")
        import traceback
        traceback.print_exc()

    print()


def test_help_functions():
    """测试帮助功能"""
    print("🧪 测试帮助功能")
    print("=" * 50)

    try:
        from claude_command_integration import get_help_info
        help_text = get_help_info()

        if help_text and len(help_text) > 100:
            print("✅ 帮助信息生成成功")
            print("📄 帮助信息长度:", len(help_text), "字符")
        else:
            print("❌ 帮助信息不完整")

    except Exception as e:
        print(f"❌ 帮助功能测试失败: {e}")

    print()


def main_test():
    """主测试函数"""
    print("🚀 开始微博热搜斜杠命令功能测试")
    print("=" * 60)
    print()

    # 运行各项测试
    test_config_parser()
    test_config_validation()
    test_help_functions()
    test_command_integration()

    print("🎉 测试完成!")
    print("\n📋 测试总结:")
    print("  - 配置解析功能: 已测试")
    print("  - 配置验证功能: 已测试")
    print("  - 帮助功能: 已测试")
    print("  - 命令集成功能: 已测试")
    print("\n✅ 斜杠命令功能基本可用!")


if __name__ == "__main__":
    main_test()